package com.digiwin.athena.schedulemanager.core.config;

import java.util.HashMap;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.core.Binding;
import org.springframework.amqp.core.BindingBuilder;
import org.springframework.amqp.core.DirectExchange;
import org.springframework.amqp.core.Queue;
import org.springframework.amqp.rabbit.connection.ConnectionFactory;
import org.springframework.amqp.rabbit.connection.CorrelationData;
import org.springframework.amqp.rabbit.core.RabbitAdmin;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.amqp.support.converter.Jackson2JsonMessageConverter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

/**
 * @author zhangww
 * @description: RabbitMQ连接
 * @date 2021/6/8 12:20
 */
@Configuration("scheduleRabbitMQConfig")
public class RabbitMQConfig implements RabbitTemplate.ConfirmCallback {

    private static final Logger LOGGER = LoggerFactory.getLogger(RabbitMQConfig.class);

    @Autowired
    @Qualifier("scheduleRabbitMQProperty")
    private RabbitMQProperty rabbitMQProperty;

    @Bean("scheduleRabbitAdmin")
    public RabbitAdmin rabbitAdmin(ConnectionFactory connectionFactory) {
        return new RabbitAdmin(connectionFactory);
    }

    @Bean(name = "scheduleRabbitTemplate")
    public RabbitTemplate rabbitTemplate(ConnectionFactory rabbitConnectionFactory) {
        RabbitTemplate rabbitTemplate = new RabbitTemplate(rabbitConnectionFactory);
        rabbitTemplate.setMessageConverter(new Jackson2JsonMessageConverter());
        rabbitTemplate.setConfirmCallback(this);
        return rabbitTemplate;
    }

    @Bean("scheduleGetDirectExchange")
    public DirectExchange getDirectExchange() {
        return new DirectExchange(rabbitMQProperty.getExchange(), true, false, null);
    }

    @Bean("scheduleGetQueue")
    public Queue getQueue(@Qualifier("scheduleRabbitAdmin") RabbitAdmin rabbitAdmin) {
        Map<String, Object> arguments = new HashMap<>(4);
        arguments.put("x-message-ttl", rabbitMQProperty.getTtl());
        arguments.put("x-dead-letter-exchange", rabbitMQProperty.getDdlExchange());
        arguments.put("x-dead-letter-routing-key", rabbitMQProperty.getDdlRoutingKey());
        Queue queue = new Queue(rabbitMQProperty.getQueue(), true, false, false, arguments);
        rabbitAdmin.declareQueue(queue);
        return queue;
    }

    @Bean("scheduleGetQueuePtm")
    public Queue getQueuePtm(@Qualifier("scheduleRabbitAdmin") RabbitAdmin rabbitAdmin) {
        Map<String, Object> arguments = new HashMap<>(4);
        arguments.put("x-message-ttl", rabbitMQProperty.getTtl());
        Queue queue = new Queue(rabbitMQProperty.getQueuePtm(), true, false, false, arguments);
        rabbitAdmin.declareQueue(queue);
        return queue;
    }

    @Bean("scheduleGetQueueTask")
    public Queue getQueueTask(@Qualifier("scheduleRabbitAdmin") RabbitAdmin rabbitAdmin) {
        Map<String, Object> arguments = new HashMap<>(4);
        arguments.put("x-message-ttl", rabbitMQProperty.getTtl());
        Queue queue = new Queue(rabbitMQProperty.getQueueTaskEngine(), true, false, false,
            arguments);
        rabbitAdmin.declareQueue(queue);
        return queue;
    }

    @Bean("scheduleGetQueueWorkflow")
    public Queue getQueueWorkflow(@Qualifier("scheduleRabbitAdmin") RabbitAdmin rabbitAdmin) {
        Map<String, Object> arguments = new HashMap<>(4);
        arguments.put("x-message-ttl", rabbitMQProperty.getTtl());
        Queue queue = new Queue(rabbitMQProperty.getQueueWorkflow(), true, false, false, arguments);
        rabbitAdmin.declareQueue(queue);
        return queue;
    }

    @Bean("scheduleBinding")
    public Binding binding(ConnectionFactory connectionFactory) {
        return BindingBuilder.bind(getQueue(rabbitAdmin(connectionFactory))).to(getDirectExchange())
            .with(rabbitMQProperty.getRoutingKey());

    }

    @Bean("scheduleBindingPtm")
    public Binding bindingPtm(ConnectionFactory connectionFactory) {
        return BindingBuilder.bind(getQueuePtm(rabbitAdmin(connectionFactory)))
            .to(getDirectExchange()).with(rabbitMQProperty.getRoutingKeyPtm());
    }

    @Bean("scheduleBindingTask")
    public Binding bindingTask(ConnectionFactory connectionFactory) {
        return BindingBuilder.bind(getQueueTask(rabbitAdmin(connectionFactory)))
            .to(getDirectExchange()).with(rabbitMQProperty.getRoutingKeyTaskEngine());
    }

    @Bean("scheduleBindingWorkflow")
    public Binding bindingWorkflow(ConnectionFactory connectionFactory) {
        return BindingBuilder.bind(getQueueWorkflow(rabbitAdmin(connectionFactory)))
            .to(getDirectExchange()).with(rabbitMQProperty.getRoutingKeyWorkflow());
    }

    @Bean("scheduleGetDeadLetterDirectExchange")
    public DirectExchange getDeadLetterDirectExchange() {
        return new DirectExchange(rabbitMQProperty.getDdlExchange(), true, false, null);
    }

    @Bean("scheduleGetDeadLetterQueue")
    public Queue getDeadLetterQueue(@Qualifier("scheduleRabbitAdmin") RabbitAdmin rabbitAdmin) {
        Queue queue = new Queue(rabbitMQProperty.getDdlQueue(), true, false, false);
        rabbitAdmin.declareQueue(queue);
        return queue;
    }

    @Bean("scheduleBindingDeadLetter")
    public Binding bindingDeadLetter(ConnectionFactory connectionFactory) {
        return BindingBuilder.bind(getDeadLetterQueue(rabbitAdmin(connectionFactory)))
            .to(getDeadLetterDirectExchange()).with(rabbitMQProperty.getDdlRoutingKey());
    }

    @Override
    public void confirm(CorrelationData correlationData, boolean isAck, String cause) {
        //fixme 后续需要考虑消息发送失败后的异常处理
        if (!isAck) {
            LOGGER.error("消息发送失败,ID为: {},错误信息: {}", correlationData.getId(), cause);
        }
    }
}
