package com.digiwin.athena.schedulemanager.core.config;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.core.Binding;
import org.springframework.amqp.core.BindingBuilder;
import org.springframework.amqp.core.DirectExchange;
import org.springframework.amqp.core.Queue;
import org.springframework.amqp.rabbit.connection.CachingConnectionFactory;
import org.springframework.amqp.rabbit.connection.ConnectionFactory;
import org.springframework.amqp.rabbit.connection.CorrelationData;
import org.springframework.amqp.rabbit.core.RabbitAdmin;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.amqp.support.converter.Jackson2JsonMessageConverter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.util.HashMap;
import java.util.Map;

/**
 * @author zhangww
 * @description: RabbitMQ连接
 * @date 2021/6/8 12:20
 */
@Configuration("scheduleRabbitMQConfig")
public class RabbitMQConfig implements RabbitTemplate.ConfirmCallback {

    private static final Logger LOGGER = LoggerFactory.getLogger(RabbitMQConfig.class);

    @Autowired
    @Qualifier("scheduleRabbitMQProperty")
    private RabbitMQProperty rabbitMQProperty;

    @Bean("scheduleRabbitAdmin")
    public RabbitAdmin rabbitAdmin(ConnectionFactory connectionFactory) {
        return new RabbitAdmin(connectionFactory);
    }

    @Bean(name = "scheduleRabbitTemplate")
    public RabbitTemplate rabbitTemplate(ConnectionFactory rabbitConnectionFactory) {
        RabbitTemplate rabbitTemplate = new RabbitTemplate(rabbitConnectionFactory);
        rabbitTemplate.setMessageConverter(new Jackson2JsonMessageConverter());
        rabbitTemplate.setConfirmCallback(this);
        return rabbitTemplate;
    }

    @Bean("scheduleGetDirectExchange")
    public DirectExchange getDirectExchange() {
        return new DirectExchange(rabbitMQProperty.getExchange(), true, false, null);
    }

    @Bean("scheduleGetQueue")
    public Queue getQueue() {
        Map<String, Object> arguments = new HashMap<>(4);
        arguments.put("x-message-ttl", rabbitMQProperty.getTtl());
        arguments.put("x-dead-letter-exchange", rabbitMQProperty.getDdlExchange());
        arguments.put("x-dead-letter-routing-key", rabbitMQProperty.getDdlRoutingKey());
        return new Queue(rabbitMQProperty.getQueue(), true, false, false, arguments);
    }

    @Bean("scheduleGetQueuePtm")
    public Queue getQueuePtm() {
        Map<String, Object> arguments = new HashMap<>(4);
        arguments.put("x-message-ttl", rabbitMQProperty.getTtl());
        return new Queue(rabbitMQProperty.getQueuePtm(), true, false, false, arguments);
    }

    @Bean("scheduleGetQueueTask")
    public Queue getQueueTask() {
        Map<String, Object> arguments = new HashMap<>(4);
        arguments.put("x-message-ttl", rabbitMQProperty.getTtl());
        return new Queue(rabbitMQProperty.getQueueTaskEngine(), true, false, false, arguments);
    }

    @Bean("scheduleGetQueueWorkflow")
    public Queue getQueueWorkflow() {
        Map<String, Object> arguments = new HashMap<>(4);
        arguments.put("x-message-ttl", rabbitMQProperty.getTtl());
        return new Queue(rabbitMQProperty.getQueueWorkflow(), true, false, false, arguments);
    }

    @Bean("scheduleBinding")
    public Binding binding() {
        return BindingBuilder.bind(getQueue()).to(getDirectExchange()).with(rabbitMQProperty.getRoutingKey());
    }

    @Bean("scheduleBindingPtm")
    public Binding bindingPtm() {
        return BindingBuilder.bind(getQueuePtm()).to(getDirectExchange()).with(rabbitMQProperty.getRoutingKeyPtm());
    }

    @Bean("scheduleBindingTask")
    public Binding bindingTask() {
        return BindingBuilder.bind(getQueueTask()).to(getDirectExchange()).with(rabbitMQProperty.getRoutingKeyTaskEngine());
    }

    @Bean("scheduleBindingWorkflow")
    public Binding bindingWorkflow() {
        return BindingBuilder.bind(getQueueWorkflow()).to(getDirectExchange()).with(rabbitMQProperty.getRoutingKeyWorkflow());
    }

    @Bean("scheduleGetDeadLetterDirectExchange")
    public DirectExchange getDeadLetterDirectExchange() {
        return new DirectExchange(rabbitMQProperty.getDdlExchange(), true, false, null);
    }

    @Bean("scheduleGetDeadLetterQueue")
    public Queue getDeadLetterQueue() {
        return new Queue(rabbitMQProperty.getDdlQueue(), true, false, false);
    }

    @Bean("scheduleBindingDeadLetter")
    public Binding bindingDeadLetter() {
        return BindingBuilder.bind(getDeadLetterQueue()).to(getDeadLetterDirectExchange()).with(rabbitMQProperty.getDdlRoutingKey());
    }

    @Override
    public void confirm(CorrelationData correlationData, boolean isAck, String cause) {
        //fixme 后续需要考虑消息发送失败后的异常处理
        if (!isAck) {
            LOGGER.error("消息发送失败,ID为: {},错误信息: {}", correlationData.getId(), cause);
        }
    }
}
