package com.digiwin.athena.schedulemanager.core.config;

import com.digiwin.app.autoconfigure.context.DWDaoDynamicDataSourcePropertySource;
import com.digiwin.app.common.DWApplicationClassLoader;
import com.digiwin.app.dao.DWDaoImpl;
import com.digiwin.app.dao.dialect.DWMySQLDialect;
import com.digiwin.app.dao.dialect.DWSQLDialect;
import com.digiwin.app.dao.filter.DWResultSetFilterChain;
import com.digiwin.app.dao.filter.IDWResultSetFilter;
import com.digiwin.app.dao.properties.DWDaoDataSourceProperties;
import com.digiwin.app.dao.properties.DWDaoProperties;
import com.digiwin.app.dao.security.AESUtil;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import javax.sql.DataSource;
import org.apache.commons.dbutils.DWQueryTimeoutQueryRunner;
import org.apache.commons.dbutils.QueryRunner;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.jdbc.DataSourceProperties;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.bind.BindHandler;
import org.springframework.boot.context.properties.bind.BindResult;
import org.springframework.boot.context.properties.bind.Bindable;
import org.springframework.boot.context.properties.bind.Binder;
import org.springframework.boot.context.properties.bind.handler.IgnoreTopLevelConverterNotFoundBindHandler;
import org.springframework.boot.context.properties.source.ConfigurationPropertyName;
import org.springframework.boot.jdbc.DataSourceBuilder;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.EnumerablePropertySource;
import org.springframework.core.env.Environment;
import org.springframework.core.env.MapPropertySource;
import org.springframework.core.env.PropertiesPropertySource;
import org.springframework.core.env.PropertySource;
import org.springframework.jdbc.datasource.DataSourceTransactionManager;
import org.springframework.jdbc.datasource.TransactionAwareDataSourceProxy;

@Configuration
public class ScheduleCommonDBConfig {

  @Bean(name = "schedule-dw-transactionManager")
  @ConditionalOnBean(name = "schedule-dataSource")
  public DataSourceTransactionManager dataSourceTransactionManager(
      @Qualifier("schedule-proxyDataSource") DataSource dataSourceProxy) {
    DataSourceTransactionManager dataSourceTransactionManager = new DataSourceTransactionManager();
    dataSourceTransactionManager.setDataSource(dataSourceProxy);
    return dataSourceTransactionManager;
  }

  /**
   * 默認 dw-dao(dw-dao) 使用的數據源
   *
   * @param context 上下文
   * @return 默認 dao 的數據源
   */
  @Bean(name = "schedule-dataSource")
  @Conditional(ScheduleDWDefaultDWDaoEnableCondition.class)
  @ConfigurationProperties("spring.datasource.schedule")
  public DataSource defaultDWDaoDataSource(ApplicationContext context) {
    String datasourceUrl = context.getEnvironment().getProperty("spring.datasource.schedule.url");
    if (StringUtils.isBlank(datasourceUrl)) {
      String datasourceUrlType = context.getEnvironment()
          .getProperty("spring.datasource.schedule.url.type");
      String datasourceUrlHost = context.getEnvironment()
          .getProperty("spring.datasource.schedule.url.host");
      String datasourceUrlPort = context.getEnvironment()
          .getProperty("spring.datasource.schedule.url.port");
      String datasourceUrlDatabase = context.getEnvironment()
          .getProperty("spring.datasource.schedule.url.database");
      String datasourceUrlArgs = context.getEnvironment()
          .getProperty("spring.datasource.schedule.url.args");

      StringBuilder datasourceUrlCombination = new StringBuilder();
      datasourceUrlCombination.append("jdbc:" + datasourceUrlType + "://" + datasourceUrlHost);
      if (!"".equals(datasourceUrlPort)) {
        datasourceUrlCombination.append(":" + datasourceUrlPort);
      }
      datasourceUrlCombination.append("/" + datasourceUrlDatabase);
      if (!"".equals(datasourceUrlArgs)) {
        datasourceUrlCombination.append("?" + datasourceUrlArgs);
      }
      //要判斷是模組化還是簡易式兩種運行環境
      PropertySource propertySource = ((ConfigurableEnvironment) context.getEnvironment()).getPropertySources()
          .get("dap-application-propertysource");
      //簡易式
      if (propertySource instanceof EnumerablePropertySource) {
        Properties properties=new Properties();
        properties.put("spring.datasource.schedule.url",
            datasourceUrlCombination.toString());
        EnumerablePropertySource scheduleUrlPropertySource= new EnumerablePropertySource<Properties>(
            "dap-dao-schedule-dynamic-propertysource", properties) {
          @Override
          public Object getProperty(String name) {
            return this.getSource().getProperty(name);
          }
          @Override
          public String[] getPropertyNames() {
            List<String> propertyNames = new ArrayList<String>();
            Enumeration<String> sourcesKeys = (Enumeration) this.getSource().keys();
            while(sourcesKeys.hasMoreElements()){
              String key = sourcesKeys.nextElement();
              propertyNames.add(key);
            }
            return propertyNames.toArray(new String[propertyNames.size()]);
          }
        };
        ((ConfigurableEnvironment) context.getEnvironment()).getPropertySources()
            .addFirst(scheduleUrlPropertySource);
      }
    }
    Binder binder = Binder.get(context.getEnvironment());
    BindHandler handler = new IgnoreTopLevelConverterNotFoundBindHandler();
    ConfigurationPropertyName cfgPropertyName = ConfigurationPropertyName.of(
        "spring.datasource.schedule");
    BindResult<DataSourceProperties> result = binder.bind(cfgPropertyName,
        Bindable.of(DataSourceProperties.class), handler);
    DataSourceBuilder<?> dataSourceBuilder = result.get().initializeDataSourceBuilder();

    return dataSourceBuilder.build();
  }

  @Bean(name = "schedule-proxyDataSource")
  @ConditionalOnBean(name = "schedule-dataSource")
  public TransactionAwareDataSourceProxy transactionAwareDataSourceProxy(
      @Qualifier("schedule-dataSource") DataSource dataSource) {
    TransactionAwareDataSourceProxy returnDataSourceProxy = new TransactionAwareDataSourceProxy(
        dataSource);
    return returnDataSourceProxy;
  }

  @Bean(name = "schedule-queryRunner")
  @ConditionalOnBean(name = "schedule-dataSource")
  public QueryRunner queryRunner(
      @Qualifier("schedule-proxyDataSource") DataSource dataSourceProxy
  ) {

    return new DWQueryTimeoutQueryRunner(dataSourceProxy);
  }

  /***
   * 不以bena的形式注入，避免和dap冲突
   * @param environment
   * @return
   * @throws Exception
   */
  private DWSQLDialect dwSqlDialect(Environment environment) throws Exception {
    DWSQLDialect dialect = null;
    String dialectClassName = environment.getProperty("spring.datasource.schedule.dialect.class");
    if (dialectClassName == null) {
      dialect = new DWMySQLDialect();
    } else {
      Class<?> dialectClass = this.getClass().getClassLoader().loadClass(dialectClassName);
      dialect = (DWSQLDialect) dialectClass.newInstance();
    }
    return dialect;
  }

  @Bean(name = "schedule-dao")
  @ConditionalOnBean(name = "schedule-queryRunner")
  public DWDaoImpl dwDaoImpl(@Qualifier("schedule-queryRunner") QueryRunner queryRunner,
      @Qualifier("dw-datasource-properties") DWDaoDataSourceProperties dataSourceProperties,
      @Qualifier("dw-dao-properties") DWDaoProperties daoProperties,Environment environment) throws Exception {
    DWDaoImpl dao = new DWDaoImpl(queryRunner);
    dao.setDialect(dwSqlDialect(environment));
    dao.setProperties(daoProperties);
    // TODO dao操作應該要歸到DWDaoDataSetProperties，在dao裡用DWDaoProperties.getDefaultProperties()取得配置
    // Add by Ma Chao @2021-11-18 begin
    // 讀取配置信息，設定是否開啟底線轉駝峰命名方式
    boolean mapUnderscoreToCamelCase = true;
    //String settingValue = DWApplicationConfigUtils.getProperty("dwDaoUnderScoreToCamelCase", "true");
    boolean settingValue = dataSourceProperties.isDwDaoUnderScoreToCamelCase();
    // 只有顯示的聲明關閉時才不開啟
    if (!settingValue) {
      mapUnderscoreToCamelCase = false;
    }
    dao.setMapUnderscoreToCamelCase(mapUnderscoreToCamelCase);
    // Add by Ma Chao @2021-11-18 end

    //Chuck 2022-05-13 insertDefaultValueFromMetadata改用自動配置注入
    boolean insertDefaultValueFromMetadata = daoProperties.getDataSetProperties()
        .isDwdaoInsertDefaultValueFromMetadata();
    dao.setInsertDefaultValueFromMetadata(insertDefaultValueFromMetadata);

    //Chuck 2022-06-20 updateDefaultValueFromMetadata增加配置
    boolean updateDefaultValueFromMetadata = daoProperties.getDataSetProperties()
        .isDwdaoUpdateDefaultValueFromMetadata();
    dao.setUpdateDefaultValueFromMetadata(updateDefaultValueFromMetadata);

    // 2022-8-12 falcon
    dao.setQueryPaginationByObjectEnabled(
        daoProperties.getDataSetProperties().isQueryPaginationByObjectEnabled());

    // 2023-02-04 Chuck 注入result set過濾器
    List<IDWResultSetFilter> resultSetFilters = new ArrayList<IDWResultSetFilter>();
    String resultSetFilterClassNames = daoProperties.getDaoResultSetFilterProperties()
        .getResultSetFilterClassNames();
    if (resultSetFilterClassNames != null && !"".equals(resultSetFilterClassNames)) {
      String[] classNames = resultSetFilterClassNames.split(",");
      for (String className : classNames) {
        Class<?> resultSetFilterClass = DWApplicationClassLoader.getInstance().loadClass(className);
        resultSetFilters.add((IDWResultSetFilter) resultSetFilterClass.newInstance());
      }
    }
    DWResultSetFilterChain resultSetFilterChain = new DWResultSetFilterChain(resultSetFilters);
    dao.setDWResultSetFilter(resultSetFilterChain);

    return dao;
  }
}