/*
 * Decompiled with CFR 0.152.
 */
package com.uber.cadence.internal.testservice;

import com.google.common.util.concurrent.Uninterruptibles;
import com.uber.cadence.internal.testservice.LockHandle;
import com.uber.cadence.internal.testservice.SelfAdvancingTimer;
import java.sql.Timestamp;
import java.time.Duration;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.PriorityQueue;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.LongSupplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class SelfAdvancingTimerImpl
implements SelfAdvancingTimer {
    private static final Logger log = LoggerFactory.getLogger(SelfAdvancingTimerImpl.class);
    private final LongSupplier clock = () -> {
        long timeMillis = this.currentTimeMillis();
        return timeMillis;
    };
    private final Lock lock = new ReentrantLock();
    private final Condition condition = this.lock.newCondition();
    private final ThreadPoolExecutor executor = new ThreadPoolExecutor(5, 5, 1L, TimeUnit.SECONDS, new LinkedBlockingDeque<Runnable>(), r -> new Thread(r, "Timer task"));
    private long currentTime;
    private int lockCount;
    private long timeLastLocked = -1L;
    private long systemTimeLastLocked = -1L;
    private boolean emptyQueue = true;
    private final LinkedList<LockEvent> lockEvents = new LinkedList();
    private final PriorityQueue<TimerTask> tasks = new PriorityQueue<TimerTask>(Comparator.comparing(TimerTask::getExecutionTime));
    private final Thread timerPump = new Thread((Runnable)new TimerPump(), "SelfAdvancingTimer Pump");

    public SelfAdvancingTimerImpl(long initialTime) {
        this.currentTime = initialTime == 0L ? System.currentTimeMillis() : initialTime;
        this.executor.setRejectedExecutionHandler(new ThreadPoolExecutor.CallerRunsPolicy());
        this.lockTimeSkipping("SelfAdvancingTimerImpl constructor");
        this.timerPump.start();
    }

    private void updateTimeLocked() {
        if (this.lockCount > 0) {
            if (this.timeLastLocked < 0L || this.systemTimeLastLocked < 0L) {
                throw new IllegalStateException("Invalid timeLastLocked or systemTimeLastLocked");
            }
            this.currentTime = this.timeLastLocked + (System.currentTimeMillis() - this.systemTimeLastLocked);
        } else {
            TimerTask task = this.tasks.peek();
            if (task != null && task.getExecutionTime() > this.currentTime) {
                this.currentTime = task.getExecutionTime();
                log.trace("Jumping to the time of the next timer task: " + this.currentTime);
            }
        }
    }

    private long currentTimeMillis() {
        this.lock.lock();
        try {
            this.updateTimeLocked();
            long l = this.currentTime;
            return l;
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void schedule(Duration delay, Runnable task) {
        this.schedule(delay, task, "unknown");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void schedule(Duration delay, Runnable task, String taskInfo) {
        this.lock.lock();
        try {
            long executionTime = delay.toMillis() + this.currentTime;
            this.tasks.add(new TimerTask(executionTime, task, taskInfo));
            if (this.tasks.size() == 1 && this.emptyQueue) {
                this.unlockTimeSkippingLocked("schedule task for " + taskInfo);
                this.emptyQueue = false;
            }
            this.condition.signal();
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public LongSupplier getClock() {
        return this.clock;
    }

    @Override
    public LockHandle lockTimeSkipping(String caller) {
        this.lock.lock();
        try {
            LockHandle lockHandle = this.lockTimeSkippingLocked(caller);
            return lockHandle;
        }
        finally {
            this.lock.unlock();
        }
    }

    private LockHandle lockTimeSkippingLocked(String caller) {
        if (this.lockCount++ == 0) {
            this.timeLastLocked = this.currentTime;
            this.systemTimeLastLocked = System.currentTimeMillis();
        }
        LockEvent event = new LockEvent(caller, LockEventType.LOCK);
        this.lockEvents.add(event);
        return new TimerLockHandle(event);
    }

    @Override
    public void unlockTimeSkipping(String caller) {
        this.lock.lock();
        try {
            this.unlockTimeSkippingLocked(caller);
            this.condition.signal();
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void updateLocks(int count, String caller) {
        this.lock.lock();
        try {
            if (count >= 0) {
                for (int i = 0; i < count; ++i) {
                    this.lockTimeSkippingLocked("updateLocks " + caller);
                }
            } else {
                for (int i = 0; i < -count; ++i) {
                    this.unlockTimeSkippingLocked("updateLocks " + caller);
                }
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void getDiagnostics(StringBuilder result) {
        result.append("Self Advancing Timer Lock Events:\n");
        this.lock.lock();
        try {
            int lockCount = 0;
            for (LockEvent event : this.lockEvents) {
                lockCount = event.lockType == LockEventType.LOCK ? ++lockCount : --lockCount;
                result.append(new Timestamp(event.timestamp)).append("\t").append((Object)event.lockType).append("\t").append(lockCount).append("\t").append(event.caller).append("\n");
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void shutdown() {
        this.executor.shutdown();
        this.timerPump.interrupt();
        Uninterruptibles.joinUninterruptibly((Thread)this.timerPump);
    }

    private void unlockTimeSkippingLocked(String caller) {
        this.unlockTimeSkippingLockedInternal();
        this.lockEvents.add(new LockEvent(caller, LockEventType.UNLOCK));
    }

    private void unlockTimeSkippingLockedInternal() {
        if (this.lockCount == 0) {
            throw new IllegalStateException("Unbalanced lock and unlock calls");
        }
        --this.lockCount;
        if (this.lockCount == 0) {
            this.timeLastLocked = -1L;
            this.systemTimeLastLocked = -1L;
        }
    }

    private class TimerLockHandle
    implements LockHandle {
        private final LockEvent event;

        public TimerLockHandle(LockEvent event) {
            this.event = event;
        }

        @Override
        public void unlock() {
            SelfAdvancingTimerImpl.this.lock.lock();
            try {
                this.unlockFromHandleLocked();
                SelfAdvancingTimerImpl.this.condition.signal();
            }
            finally {
                SelfAdvancingTimerImpl.this.lock.unlock();
            }
        }

        private void unlockFromHandleLocked() {
            Boolean removed = SelfAdvancingTimerImpl.this.lockEvents.remove(this.event);
            if (!removed.booleanValue()) {
                throw new IllegalStateException("Unbalanced lock and unlock calls");
            }
            SelfAdvancingTimerImpl.this.unlockTimeSkippingLockedInternal();
        }
    }

    private static enum LockEventType {
        LOCK,
        UNLOCK;


        public String toString() {
            return this == LOCK ? "L" : "U";
        }
    }

    private static class LockEvent {
        String caller;
        LockEventType lockType;
        long timestamp;

        public LockEvent(String caller, LockEventType lockType) {
            this.caller = caller;
            this.lockType = lockType;
            this.timestamp = System.currentTimeMillis();
        }
    }

    private class TimerPump
    implements Runnable {
        private TimerPump() {
        }

        @Override
        public void run() {
            SelfAdvancingTimerImpl.this.lock.lock();
            try {
                this.runLocked();
            }
            catch (Throwable e) {
                log.error("Timer pump failed", e);
            }
            finally {
                SelfAdvancingTimerImpl.this.lock.unlock();
            }
        }

        private void runLocked() {
            while (!Thread.currentThread().isInterrupted()) {
                TimerTask peekedTask;
                SelfAdvancingTimerImpl.this.updateTimeLocked();
                if (!SelfAdvancingTimerImpl.this.emptyQueue && SelfAdvancingTimerImpl.this.tasks.isEmpty()) {
                    SelfAdvancingTimerImpl.this.lockTimeSkippingLocked("runLocked");
                    SelfAdvancingTimerImpl.this.emptyQueue = true;
                }
                if ((peekedTask = (TimerTask)SelfAdvancingTimerImpl.this.tasks.peek()) != null && peekedTask.getExecutionTime() <= SelfAdvancingTimerImpl.this.currentTime) {
                    try {
                        LockHandle lockHandle = SelfAdvancingTimerImpl.this.lockTimeSkippingLocked("runnable " + peekedTask.getTaskInfo());
                        TimerTask polledTask = (TimerTask)SelfAdvancingTimerImpl.this.tasks.poll();
                        Runnable runnable = polledTask.getRunnable();
                        SelfAdvancingTimerImpl.this.executor.execute(() -> {
                            try {
                                runnable.run();
                            }
                            finally {
                                lockHandle.unlock();
                            }
                        });
                    }
                    catch (Throwable e) {
                        log.error("Timer task failure", e);
                    }
                    continue;
                }
                long timeToAwait = peekedTask == null ? Long.MAX_VALUE : peekedTask.getExecutionTime() - SelfAdvancingTimerImpl.this.currentTime;
                try {
                    SelfAdvancingTimerImpl.this.condition.await(timeToAwait, TimeUnit.MILLISECONDS);
                }
                catch (InterruptedException e) {
                    break;
                }
            }
        }
    }

    private static class TimerTask {
        private final long executionTime;
        private final Runnable runnable;
        private final String taskInfo;

        TimerTask(long executionTime, Runnable runnable, String taskInfo) {
            this.executionTime = executionTime;
            this.runnable = runnable;
            this.taskInfo = taskInfo;
        }

        long getExecutionTime() {
            return this.executionTime;
        }

        public Runnable getRunnable() {
            return this.runnable;
        }

        String getTaskInfo() {
            return this.taskInfo;
        }

        public String toString() {
            return "TimerTask{executionTime=" + this.executionTime + '}';
        }
    }
}

