package com.digiwin.dap.middleware.commons.crypto;

import cn.hutool.crypto.KeyUtil;
import cn.hutool.crypto.SecureUtil;
import cn.hutool.crypto.asymmetric.AsymmetricAlgorithm;
import cn.hutool.crypto.asymmetric.KeyType;
import com.digiwin.dap.middleware.exception.CommonException;
import com.digiwin.dap.middleware.commons.core.codec.Base64;
import com.digiwin.dap.middleware.commons.crypto.constant.AlgorithmEnum;
import com.digiwin.dap.middleware.commons.crypto.constant.TransformationEnum;
import com.digiwin.dap.middleware.commons.util.StrUtils;

import javax.crypto.Cipher;
import javax.crypto.spec.OAEPParameterSpec;
import javax.crypto.spec.PSource;
import java.security.*;
import java.security.spec.MGF1ParameterSpec;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Objects;

/**
 * 非对称加密算法
 * <pre>
 * 1、签名：使用私钥加密，公钥解密。
 * 用于让所有公钥所有者验证私钥所有者的身份并且用来防止私钥所有者发布的内容被篡改，但是不用来保证内容不被他人获得。
 *
 * 2、加密：用公钥加密，私钥解密。
 * 用于向公钥所有者发布信息,这个信息可能被他人篡改,但是无法被他人获得。
 * </pre>
 *
 * <p>
 * RSA公钥/私钥/签名加密解密
 * <p>
 * 罗纳德·李维斯特（Ron [R]ivest）、阿迪·萨莫尔（Adi [S]hamir）和伦纳德·阿德曼（Leonard [A]dleman）
 * <p>
 * 由于非对称加密速度极其缓慢，一般文件不使用它来加密而是使用对称加密，<br>
 * 非对称加密算法可以用来对对称加密的密钥加密，这样保证密钥的安全也就保证了数据的安全
 *
 * @author fobgochod
 * @since 1.0.0
 */
public class RSA {

    private static final cn.hutool.crypto.asymmetric.RSA rsa_ = SecureUtil.rsa();

    private RSA() {
    }

    /**
     * ECB加密
     *
     * @param data            被加密的bytes
     * @param publicKeyBase64 Base64编码后的公钥
     * @return 加密后的String
     */
    public static String encrypt(String data, String publicKeyBase64) {
        PublicKey publicKey = KeyUtil.generatePublicKey(AsymmetricAlgorithm.RSA_ECB_PKCS1.getValue(), Base64.decode(publicKeyBase64));
        byte[] bytes = rsa_.setPublicKey(publicKey).encrypt(data, KeyType.PublicKey);
        return Base64.encode(bytes);
    }

    /**
     * ECB解密
     *
     * @param data             被解密的bytes
     * @param privateKeyBase64 Base64编码后的私钥
     * @return 解密后的String
     */
    public static String decrypt(String data, String privateKeyBase64) {
        PrivateKey privateKey = KeyUtil.generatePrivateKey(AsymmetricAlgorithm.RSA_ECB_PKCS1.getValue(), Base64.decode(privateKeyBase64));
        byte[] bytes = rsa_.setPrivateKey(privateKey).decrypt(data, KeyType.PrivateKey);
        return StrUtils.str(bytes);
    }

    /**
     * OAEP加密
     *
     * @param data            被加密的bytes
     * @param publicKeyBase64 Base64编码后的公钥
     * @return 加密后的bytes
     */
    public static byte[] encryptOAEP(byte[] data, String publicKeyBase64) {
        byte[] bytes = Base64.decode(publicKeyBase64);
        return encrypt(data, bytes, TransformationEnum.RSA_ECB_OAEP.getValue());
    }

    /**
     * OAEP解密
     *
     * @param data             被解密的bytes
     * @param privateKeyBase64 Base64编码后的私钥
     * @return 解密后的bytes
     */
    public static byte[] decryptOAEP(byte[] data, String privateKeyBase64) {
        byte[] bytes = Base64.decode(privateKeyBase64);
        return decrypt(data, bytes, TransformationEnum.RSA_ECB_OAEP.getValue());
    }

    /**
     * RSA-OAEP加密
     *
     * @param data           被加密的bytes
     * @param publicKey      公钥
     * @param transformation 算法/工作模式/填充模式
     * @return 加密后的bytes
     */
    public static byte[] encrypt(byte[] data, byte[] publicKey, String transformation) {
        try {
            X509EncodedKeySpec keySpec = new X509EncodedKeySpec(publicKey);
            KeyFactory keyFactory = KeyFactory.getInstance(AlgorithmEnum.RSA.getValue());
            PublicKey key = keyFactory.generatePublic(keySpec);
            Cipher cipher = Cipher.getInstance(transformation);
            if (Objects.equals(transformation, TransformationEnum.RSA_ECB_OAEP.getValue())) {
                OAEPParameterSpec params = new OAEPParameterSpec("SHA-256", "MGF1", new MGF1ParameterSpec("SHA-1"), PSource.PSpecified.DEFAULT);
                cipher.init(Cipher.ENCRYPT_MODE, key, params);
            } else {
                cipher.init(Cipher.ENCRYPT_MODE, key);
            }
            return cipher.doFinal(data);
        } catch (Exception e) {
            throw new CommonException("RSA加密异常", e);
        }
    }


    /**
     * RSA-OAEP解密
     *
     * @param data           被解密的bytes
     * @param privateKey     私钥
     * @param transformation 算法/工作模式/填充模式
     * @return 解密后的bytes
     */
    public static byte[] decrypt(byte[] data, byte[] privateKey, String transformation) {
        try {
            PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(privateKey);
            KeyFactory keyFactory = KeyFactory.getInstance(AlgorithmEnum.RSA.getValue());
            PrivateKey key = keyFactory.generatePrivate(keySpec);
            Cipher cipher = Cipher.getInstance(transformation);
            if (Objects.equals(transformation, TransformationEnum.RSA_ECB_OAEP.getValue())) {
                OAEPParameterSpec oaepParams = new OAEPParameterSpec("SHA-256", "MGF1", new MGF1ParameterSpec("SHA-1"), PSource.PSpecified.DEFAULT);
                cipher.init(Cipher.DECRYPT_MODE, key, oaepParams);
            } else {
                cipher.init(Cipher.DECRYPT_MODE, key);
            }
            return cipher.doFinal(data);
        } catch (Exception e) {
            throw new CommonException("RSA解密异常", e);
        }
    }

    /**
     * 生成RSA密钥对
     *
     * @param keySize 密钥模（modulus）长度（单位bit）
     * @return 密钥对
     */
    public static KeyPair generateKeyPair(int keySize) {
        try {
            KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance(AlgorithmEnum.RSA.getValue());
            keyPairGenerator.initialize(keySize);
            return keyPairGenerator.generateKeyPair();
        } catch (Exception e) {
            throw new CommonException("获取RSA密钥异常,密钥长度:" + keySize, e);
        }
    }
}
