package com.digiwin.dap.middleware.commons.util;

import com.digiwin.dap.middleware.exception.CommonException;
import com.fasterxml.jackson.annotation.JsonIgnore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.*;
import java.util.*;

/**
 * BeanUtils
 *
 * @author fobgochod
 * @since 1.0.12
 */
public class BeanUtils {

    private static final Logger logger = LoggerFactory.getLogger(BeanUtils.class);

    private BeanUtils() {
    }

    /**
     * 获取对象所有字段
     * <p>
     * 排除合成字段
     * 为了收集数据，JaCoCo会植入私有静态$jacocoData变量和私有静态方法$jacocoInit()，
     * 如果使用反射循环成员变量进而拼凑set或者get方法[get$jacocoData()]，会产生NoSuchMethodException，
     * 建议在循环中利用isSynthetic()方法检查成员变量
     * <p>
     * 排除static字段
     * 引入日志框架异常
     * private static final Logger logger
     *
     * @param object 对象
     * @return 字段
     */
    public static List<Field> getFields(Object object) {
        List<Field> fields = new ArrayList<>();
        Class<?> aClass = object.getClass();
        // 当父类为null的时候说明到达了最上层的父类(Object类).
        while (aClass != null) {
            Field[] declaredFields = aClass.getDeclaredFields();
            Arrays.stream(declaredFields).filter(field -> !field.isSynthetic() && !Modifier.isStatic(field.getModifiers())).forEach(fields::add);
            // 得到父类,然后赋给自己
            aClass = aClass.getSuperclass();
        }
        return fields;
    }

    /**
     * 将origin属性注入到destination中
     *
     * @param origin      源对象
     * @param destination 目标对象
     */
    public static void mergeObject(Object origin, Object destination) {

        if (origin == null || destination == null) {
            return;
        }
        if (!origin.getClass().equals(destination.getClass())) {
            return;
        }
        List<Field> fieldList = getFields(origin);

        for (Field field : fieldList) {
            field.setAccessible(true);
            try {
                Object value = field.get(origin);
                if (null != value) {
                    field.set(destination, value);
                }
                field.setAccessible(false);
            } catch (IllegalAccessException e) {
                throw new CommonException("对象合并错误！", e);
            }
        }
    }

    /**
     * 将origin属性注入到destination中
     *
     * @param origin      源对象
     * @param destination 目标对象
     */
    public static void mergeDifferentTypeObject(Object origin, Object destination) {
        if (origin == null || destination == null) {
            return;
        }
        List<Field> originFieldList = getFields(origin);
        List<Field> destinationFieldList = getFields(destination);

        for (Field origField : originFieldList) {
            for (Field destField : destinationFieldList) {
                if (!origField.getName().equals(destField.getName())) {
                    continue;
                }
                if (origField.getAnnotation(JsonIgnore.class) == null) {
                    mergeField(origField, destField, origin, destination);
                }
            }
        }
    }

    private static void mergeField(Field origField, Field destField, Object origin, Object destination) {
        try {
            origField.setAccessible(true);
            Object value = origField.get(origin);
            origField.setAccessible(false);

            if (value != null) {
                destField.setAccessible(true);
                destField.set(destination, value);
                destField.setAccessible(false);
            }
        } catch (IllegalAccessException e) {
            throw new CommonException("对象复制错误", e);
        }
    }


    /**
     * 循环向上转型, 获取对象的 DeclaredMethod
     *
     * @param object         : 子类对象
     * @param methodName     : 父类中的方法名
     * @param parameterTypes : 父类中的方法参数类型
     * @return 父类中的方法对象
     */

    public static Method getDeclaredMethod(Object object, String methodName, Class<?>... parameterTypes) {
        Method method = null;

        for (Class<?> clazz = object.getClass(); clazz != Object.class; clazz = clazz.getSuperclass()) {
            try {
                method = clazz.getDeclaredMethod(methodName, parameterTypes);
                return method;
            } catch (NoSuchMethodException | SecurityException e) {
                // 这里甚么都不要做！并且这里的异常必须这样写，不能抛出去。
                // 如果这里的异常打印或者往外抛，则就不会执行clazz = clazz.getSuperclass(),最后就不会进入到父类中了
                logger.error(e.getMessage(), e);
            }
        }

        return null;
    }

    /**
     * 直接调用对象方法, 而忽略修饰符(private, protected, default)
     *
     * @param object         : 子类对象
     * @param methodName     : 父类中的方法名
     * @param parameterTypes : 父类中的方法参数类型
     * @param parameters     : 父类中的方法参数
     * @return 父类中方法的执行结果
     */

    public static Object invokeMethod(Object object, String methodName, Class<?>[] parameterTypes,
                                      Object[] parameters) {
        // 根据 对象、方法名和对应的方法参数 通过反射 调用上面的方法获取 Method 对象
        Method method = getDeclaredMethod(object, methodName, parameterTypes);
        try {
            if (method != null) {
                // 抑制Java对方法进行检查,主要是针对私有方法而言
                method.setAccessible(true);  // WARNING: Reflective access to private fields
                // 调用object 的 method 所代表的方法，其方法的参数是 parameters
                return method.invoke(object, parameters);
            }
        } catch (IllegalArgumentException | IllegalAccessException | InvocationTargetException e) {
            logger.error(e.getMessage(), e);
        }
        return null;
    }

    /**
     * 循环向上转型, 获取对象上所有字段
     *
     * @param clazz :  类型
     * @return 字段
     */
    public static List<Field> getDeclaredFields(Class<?> clazz) {
        List<Field> result = new ArrayList<>();

        Field[] fields = clazz.getDeclaredFields();
        Collections.addAll(result, fields);
        Class<?> superClass = clazz.getSuperclass();
        if (superClass == Object.class) {
            return result;
        }
        List<Field> superFields = getDeclaredFields(superClass);
        result.addAll(superFields);
        return result;
    }

    /**
     * 循环向上转型, 获取对象的所有字段
     *
     * @param clazz     : 类型
     * @param fieldName : 属性名
     * @return 类中的所有属性对象
     */

    public static Field getDeclaredField(Class<?> clazz, String fieldName) {
        Field field;
        try {
            field = clazz.getDeclaredField(fieldName);
            return field;
        } catch (NoSuchFieldException e) {
            // 继续操作父类
        }

        Class<?> superClass = clazz.getSuperclass();
        if (superClass == Object.class) {
            return null;
        }
        try {
            field = getDeclaredField(superClass, fieldName);
            if (field != null) {
                return field;
            }
        } catch (Exception e) {
            // 这里甚么都不要做！并且这里的异常必须这样写，不能抛出去。
            // 如果这里的异常打印或者往外抛，则就不会执行clazz = clazz.getSuperclass(),最后就不会进入到父类中了
        }
        return null;
    }

    /**
     * 直接设置对象属性值, 忽略 private/protected 修饰符, 也不经过 setter
     *
     * @param object    : 子类对象
     * @param fieldName : 父类中的属性名
     * @param value     : 将要设置的值
     */

    public static void setFieldValue(Object object, String fieldName, Object value) {
        try {
            // 根据 对象和属性名通过反射 调用上面的方法获取 Field对象
            Field field = getDeclaredField(object.getClass(), fieldName);
            if (field != null) {
                // 抑制Java对其的检查
                field.setAccessible(true);  // WARNING: Reflective access to private fields
                // 将 object 中 field 所代表的值 设置为 value
                field.set(object, value);
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
    }

    /**
     * 直接读取对象的属性值, 忽略 private/protected 修饰符, 也不经过 getter
     *
     * @param object    : 子类对象
     * @param fieldName : 父类中的属性名
     * @return : 父类中的属性值
     */

    public static Object getFieldValue(Object object, String fieldName) {
        try {
            // 根据 对象和属性名通过反射 调用上面的方法获取 Field对象
            Field field = getDeclaredField(object.getClass(), fieldName);
            if (field != null) {
                // 抑制Java对其的检查
                field.setAccessible(true); // WARNING: Reflective access to private fields
                // 获取 object 中 field 所代表的属性值
                return field.get(object);
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
        return null;
    }

    /**
     * 将数组数据转换为实体类
     * 此处数组元素的顺序必须与实体类构造函数中的属性顺序一致
     *
     * @param list  数组对象集合
     * @param clazz 实体类
     * @param <T>   实体类
     * @param model 实例化的实体类
     * @return 实体类集合
     */
    public static <T> List<T> castEntity(List<Object[]> list, Class<T> clazz, Object model) {
        List<T> returnList = new ArrayList<>();
        if (list.isEmpty()) {
            return returnList;
        }
        // 获取每个数组集合的元素个数
        Object[] co = list.get(0);

        // 获取当前实体类的属性名、属性值、属性类别
        List<Map<String, Object>> attributeInfoList = getFieldsInfo(model);
        // 创建属性类别数组
        Class[] c2 = new Class[attributeInfoList.size()];
        // 如果数组集合元素个数与实体类属性个数不一致则发生错误
        if (attributeInfoList.size() != co.length) {
            return returnList;
        }
        // 确定构造方法
        for (int i = 0; i < attributeInfoList.size(); i++) {
            c2[i] = (Class) attributeInfoList.get(i).get("type");
        }
        try {
            for (Object[] o : list) {
                Constructor<T> constructor = clazz.getConstructor(c2);
                returnList.add(constructor.newInstance(o));
            }
        } catch (Exception ex) {
            logger.error("实体数据转化为实体类发生异常：异常信息：{}", ex.getMessage());
            return returnList;
        }
        return returnList;
    }

    /**
     * 根据属性名获取属性值
     *
     * @param fieldName 属性名
     * @param modle     实体类
     * @return 属性值
     */
    private static Object getFieldValueByName(String fieldName, Object modle) {
        try {
            String firstLetter = fieldName.substring(0, 1).toUpperCase();
            String getter = "get" + firstLetter + fieldName.substring(1);
            Method method = modle.getClass().getMethod(getter);
            return method.invoke(modle);
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 获取属性类型(type)，属性名(name)，属性值(value)的map组成的list
     *
     * @param model 实体类
     * @return list集合
     */
    private static List<Map<String, Object>> getFieldsInfo(Object model) {
        Field[] fields = model.getClass().getDeclaredFields();
        List<Map<String, Object>> list = new ArrayList<>(fields.length);
        for (Field field : fields) {
            Map<String, Object> infoMap = new HashMap<>(3);
            infoMap.put("type", field.getType());
            infoMap.put("name", field.getName());
            infoMap.put("value", getFieldValueByName(field.getName(), model));
            list.add(infoMap);
        }
        return list;
    }

}
