package com.digiwin.dap.middleware.util;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.*;
import java.time.chrono.ChronoLocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAccessor;
import java.util.Date;

/**
 * 日期工具类
 *
 * @author fobgochod
 * @author chenzhuang
 * @since 1.0.12
 */
public class DateUtils {

    /**
     * 日期时间格式：yyyy-MM-dd HH:mm:ss
     */
    public static final String PATTERN_DATETIME = "yyyy-MM-dd HH:mm:ss";
    /**
     * 日期格式：yyyy-MM-dd
     */
    public static final String PATTERN_DATE = "yyyy-MM-dd";
    /**
     * 时间格式：HH:mm:ss
     */
    public static final String PATTERN_TIME = "HH:mm:ss";
    /**
     * 日期时间格式：yyyyMMddHHmmss
     */
    public static final String PATTERN_PURE_DATETIME = "yyyyMMddHHmmss";
    /**
     * 日期格式：yyyyMMdd
     */
    public static final String PATTERN_PURE_DATE = "yyyyMMdd";
    /**
     * 时间格式：HHmmss
     */
    public static final String PATTERN_PURE_TIME = "HHmmss";
    /**
     * 日期时间格式：yyyy-MM-dd HH:mm:ss.SSS
     */
    public static final String PATTERN_PATTERN_NORM_DATETIME_MS = "yyyy-MM-dd HH:mm:ss.SSS";
    /**
     * 日期时间格式：yyyyMMddHHmmssSSS
     */
    public static final String PATTERN_PURE_DATETIME_MS = "yyyyMMddHHmmssSSS";

    public static LocalDateTime getDateByStr(String dateStr) {
        return getDateByStr(dateStr, PATTERN_DATETIME);
    }

    public static LocalDateTime getDateByStr(String dateStr, String formatter) {
        if (dateStr == null) {
            return null;
        }
        DateTimeFormatter dtf = DateTimeFormatter.ofPattern(formatter);
        return LocalDateTime.parse(dateStr, dtf);
    }


    /**
     * 当前时间，默认时区
     *
     * @return {@link LocalDateTime}
     */
    public static LocalDateTime now() {
        return LocalDateTime.now();
    }

    // ------------------------------------ Format ----------------------------------------------

    /**
     * Date 日期时间格式化
     * 格式 yyyy-MM-dd HH:mm:ss
     *
     * @param date 时间
     * @return 格式化后的时间
     */
    public static String formatDateTime(Date date) {
        return format(date, PATTERN_DATETIME);
    }

    /**
     * Date 日期格式化
     * 格式 yyyy-MM-dd
     *
     * @param date 时间
     * @return 格式化后的时间
     */
    public static String formatDate(Date date) {
        return format(date, PATTERN_DATE);
    }

    /**
     * Date 时间格式化
     * 格式 HH:mm:ss
     *
     * @param date 时间
     * @return 格式化后的时间
     */
    public static String formatTime(Date date) {
        return format(date, PATTERN_TIME);
    }

    /**
     * Date 日期格式化
     *
     * @param date    时间
     * @param pattern 表达式
     * @return 格式化后的时间
     */
    public static String format(Date date, String pattern) {
        return new SimpleDateFormat(pattern).format(date);
    }

    /**
     * java8 日期时间格式化
     * 格式 yyyy-MM-dd HH:mm:ss
     *
     * @param temporal 时间
     * @return 格式化后的时间
     */
    public static String formatDateTime(TemporalAccessor temporal) {
        return format(temporal, PATTERN_DATETIME);
    }

    /**
     * java8 日期格式化
     * 格式 yyyy-MM-dd
     *
     * @param temporal 时间
     * @return 格式化后的时间
     */
    public static String formatDate(TemporalAccessor temporal) {
        return format(temporal, PATTERN_DATE);
    }

    /**
     * java8 时间格式化
     * 格式 HH:mm:ss
     *
     * @param temporal 时间
     * @return 格式化后的时间
     */
    public static String formatTime(TemporalAccessor temporal) {
        return format(temporal, PATTERN_TIME);
    }

    /**
     * java8 日期时间格式化
     * 格式 yyyyMMddHHmmss
     *
     * @param temporal 时间
     * @return 格式化后的时间
     */
    public static String formatDateTimePure(TemporalAccessor temporal) {
        return format(temporal, PATTERN_PURE_DATETIME);
    }

    /**
     * java8 日期时间格式化
     * 格式 yyyyMMdd
     *
     * @param temporal 时间
     * @return 格式化后的时间
     */
    public static String formatDatePure(TemporalAccessor temporal) {
        return format(temporal, PATTERN_PURE_DATE);
    }

    /**
     * java8 时间格式化
     * 格式 HHmmss
     *
     * @param temporal 时间
     * @return 格式化后的时间
     */
    public static String formatTimePure(TemporalAccessor temporal) {
        return format(temporal, PATTERN_PURE_TIME);
    }

    /**
     * java8 日期时间格式化（含毫秒）
     * 格式 yyyy-MM-dd HH:mm:ss.SSS
     *
     * @param temporal 时间
     * @return 格式化后的时间
     */
    public static String formatDateTimeMsNorm(TemporalAccessor temporal) {
        return format(temporal, PATTERN_PATTERN_NORM_DATETIME_MS);
    }

    /**
     * java8 日期时间格式化（含毫秒）
     * 格式 yyyyMMddHHmmssSSS
     *
     * @param temporal 时间
     * @return 格式化后的时间
     */
    public static String formatDateTimeMsPure(TemporalAccessor temporal) {
        return format(temporal, PATTERN_PURE_DATETIME_MS);
    }

    /**
     * java8 日期格式化
     *
     * @param temporal 时间
     * @param pattern  表达式
     * @return 格式化后的时间
     */
    public static String format(TemporalAccessor temporal, String pattern) {
        return DateTimeFormatter.ofPattern(pattern).format(temporal);
    }

    // ------------------------------------ Parse ----------------------------------------------

    /**
     * 解析日期时间字符串
     *
     * @param dateStr 时间字符串
     * @return 时间
     */
    public static Date parseDateTime(String dateStr) {
        return parse(dateStr, PATTERN_DATETIME);
    }

    /**
     * 解析日期时间字符串
     * 格式 yyyy-MM-dd
     *
     * @param dateStr 时间字符串
     * @return 时间
     */
    public static Date parseDate(String dateStr) {
        return parse(dateStr, PATTERN_DATE);
    }

    /**
     * 解析日期时间字符串
     * 格式 HH:mm:ss
     *
     * @param dateStr 时间字符串
     * @return 时间
     */
    public static Date parseTime(String dateStr) {
        return parse(dateStr, PATTERN_TIME);
    }

    /**
     * 将字符串转换为时间
     *
     * @param dateStr 时间字符串
     * @param pattern 表达式
     * @return 时间
     */
    public static Date parse(String dateStr, String pattern) {
        return parse(dateStr, new SimpleDateFormat(pattern));
    }

    /**
     * 将字符串转换为时间
     *
     * @param dateStr 时间字符串
     * @param format  ConcurrentDateFormat
     * @return 时间
     */
    public static Date parse(String dateStr, DateFormat format) {
        try {
            return format.parse(dateStr);
        } catch (ParseException e) {
            throw new RuntimeException("日期时间转化失败", e);
        }
    }

    /**
     * 解析日期时间字符串
     * 格式 yyyy-MM-dd HH:mm:ss
     *
     * @param dateStr 时间字符串
     * @return 时间
     */
    public static LocalDateTime parseLocalDateTime(String dateStr) {
        return parseLocalDateTime(dateStr, PATTERN_DATETIME);
    }

    /**
     * 解析日期字符串
     * 格式 yyyy-MM-dd
     *
     * @param dateStr 时间字符串
     * @return 时间
     */
    public static LocalDateTime parseLocalDate(String dateStr) {
        return parseLocalDateTime(dateStr, PATTERN_DATE);
    }

    /**
     * 解析时间字符串
     * 格式 HH:mm:ss
     *
     * @param dateStr 时间字符串
     * @return 时间
     */
    public static LocalDateTime parseLocalTime(String dateStr) {
        return parseLocalDateTime(dateStr, PATTERN_TIME);
    }

    /**
     * 将字符串转换为时间
     *
     * @param dateStr 时间字符串
     * @param pattern 表达式
     * @return 时间
     */
    public static LocalDateTime parseLocalDateTime(String dateStr, String pattern) {
        return parseLocalDateTime(dateStr, DateTimeFormatter.ofPattern(pattern));
    }

    /**
     * 将字符串转换为时间
     *
     * @param dateStr   时间字符串
     * @param formatter DateTimeFormatter
     * @return 时间
     */
    public static LocalDateTime parseLocalDateTime(String dateStr, DateTimeFormatter formatter) {
        return LocalDateTime.parse(dateStr, formatter);
    }

    /**
     * LocalDateTime 时间转 Instant
     *
     * @param dateTime 时间
     * @return Instant
     */
    public static Instant toInstant(LocalDateTime dateTime) {
        return dateTime.atZone(ZoneId.systemDefault()).toInstant();
    }

    /**
     * Instant 转 时间 LocalDateTime
     *
     * @param instant Instant
     * @return Instant
     */
    public static LocalDateTime toDateTime(Instant instant) {
        return LocalDateTime.ofInstant(instant, ZoneId.systemDefault());
    }

    /**
     * LocalDateTime 转换成 date
     *
     * @param dateTime LocalDateTime
     * @return Date
     */
    public static Date toDate(LocalDateTime dateTime) {
        return Date.from(toInstant(dateTime));
    }

    /**
     * LocalDate 转换成 date
     *
     * @param localDate LocalDate
     * @return Date
     */
    public static Date toDate(final LocalDate localDate) {
        return Date.from(localDate.atStartOfDay(ZoneId.systemDefault()).toInstant());
    }

    /**
     * localDateTime 转换成毫秒数
     *
     * @param localDateTime LocalDateTime
     * @return long
     */
    public static long toMilliseconds(final LocalDateTime localDateTime) {
        return localDateTime.atZone(ZoneId.systemDefault()).toInstant().toEpochMilli();
    }

    /**
     * localDate 转换成毫秒数
     *
     * @param localDate LocalDate
     * @return long
     */
    public static long toMilliseconds(LocalDate localDate) {
        return toMilliseconds(localDate.atStartOfDay());
    }

    /**
     * date 转换成java8 时间LocalDateTime
     *
     * @param date Date
     * @return LocalDateTime
     */
    public static LocalDateTime toDateTime(final Date date) {
        return LocalDateTime.ofInstant(date.toInstant(), ZoneId.systemDefault());
    }

    /**
     * milliseconds 转换成 java8 时间LocalDateTime
     *
     * @param milliseconds 毫秒数
     * @return LocalDateTime
     */
    public static LocalDateTime toDateTime(final long milliseconds) {
        return LocalDateTime.ofInstant(Instant.ofEpochMilli(milliseconds), ZoneId.systemDefault());
    }

    /**
     * 获取两个日期的差，如果结束时间早于开始时间，获取结果为负。
     * <p>
     * 返回结果为{@link Duration}对象，通过调用toXXX方法返回相差单位
     *
     * @param startTimeInclude 开始时间（包含）
     * @param endTimeExclude   结束时间（不包含）
     * @return 时间差 {@link Duration}对象
     */
    public static Duration between(LocalDateTime startTimeInclude, LocalDateTime endTimeExclude) {
        return Duration.between(startTimeInclude, endTimeExclude);
    }

    /**
     * 获取两个日期的表象时间差，如果结束时间早于开始时间，获取结果为负。年月日为单位
     * <p>
     * 比如2011年2月1日，和2021年8月11日，日相差了10天，月相差6月
     *
     * @param startTimeInclude 开始时间（包括）
     * @param endTimeExclude   结束时间（不包括）
     * @return 时间差
     * @since 5.4.5
     */
    public static Period between(LocalDate startTimeInclude, LocalDate endTimeExclude) {
        return Period.between(startTimeInclude, endTimeExclude);
    }

    /**
     * 比较2个 时间差
     *
     * @param startDate 开始时间
     * @param endDate   结束时间
     * @return 时间间隔
     */
    public static Duration between(Date startDate, Date endDate) {
        return Duration.between(startDate.toInstant(), endDate.toInstant());
    }

    /**
     * 修改为一天的开始时间，例如：2020-02-02 00:00:00,000
     *
     * @param time 日期时间
     * @return 一天的开始时间
     */
    public static LocalDateTime beginOfDay(LocalDateTime time) {
        return time.with(LocalTime.MIN);
    }

    /**
     * 修改为一天的结束时间，例如：2020-02-02 23:59:59
     *
     * @param time 日期时间
     * @return 一天的结束时间
     */
    public static LocalDateTime endOfDay(LocalDateTime time) {
        return time.with(LocalTime.of(23, 59, 59));
    }

    /**
     * 是否为周末（周六或周日）
     *
     * @param localDateTime 判定的日期{@link LocalDateTime}
     * @return 是否为周末（周六或周日）
     */
    public static boolean isWeekend(LocalDateTime localDateTime) {
        return isWeekend(localDateTime.toLocalDate());
    }

    /**
     * 是否为周末（周六或周日）
     *
     * @param localDate 判定的日期{@link LocalDate}
     * @return 是否为周末（周六或周日）
     */
    public static boolean isWeekend(LocalDate localDate) {
        final DayOfWeek dayOfWeek = localDate.getDayOfWeek();
        return DayOfWeek.SATURDAY == dayOfWeek || DayOfWeek.SUNDAY == dayOfWeek;
    }

    /**
     * 检查两个时间段是否有时间重叠<br>
     * 重叠指两个时间段是否有交集
     *
     * @param realStartTime 第一个时间段的开始时间
     * @param realEndTime   第一个时间段的结束时间
     * @param startTime     第二个时间段的开始时间
     * @param endTime       第二个时间段的结束时间
     * @return true 表示时间有重合
     */
    public static boolean isOverlap(ChronoLocalDateTime<?> realStartTime, ChronoLocalDateTime<?> realEndTime,
                                    ChronoLocalDateTime<?> startTime, ChronoLocalDateTime<?> endTime) {
        return startTime.isAfter(realEndTime) || endTime.isBefore(realStartTime);
    }
}
