package com.digiwin.dap.middleware.util;

import com.digiwin.dap.middleware.exception.CommonException;

import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Arrays;
import java.util.Date;

/**
 * 雪花ID算法改造，16位大约可用31年(2025.1~2055.12) <br>
 * 机器码3位，0-7，最多8台机器 <br>
 * 序号10位，0-1023，一毫秒最多生成1024个ID <br>
 * <p>
 * JS中Number类型有个最大值（安全值），为9007199254740992，是2^53。如果超过这个值，那么js会出现不精确的问题。<br>
 * 输入： Number.MAX_SAFE_INTEGER <br>
 * 输出： 9007199254740991(2^53-1) <br>
 * <p>
 * 所以主键最大长度只能是16位<br>
 *
 * @author fobgochod
 */
public class SnowFlake {

    /**
     * 起始的时间戳 2021-02-18 03:34:47
     * 一年毫秒数 31536000000
     * long year = 3600 * 24 * 365 * 1000L;
     */
    private static final long START_STAMP = 1613590487000L;
    /**
     * 每一部分占用的位数
     * SEQUENCE_BIT 序列号占用的位数
     * MACHINE_BIT 机器标识占用的位数
     */
    private static final long SEQUENCE_BIT = 10;
    private static final long MACHINE_BIT = 3;
    /**
     * 每一部分的最大值
     */
    private static final long MAX_SEQUENCE = ~(-1L << SEQUENCE_BIT);
    private static final long MAX_MACHINE_NUM = ~(-1L << MACHINE_BIT);
    /**
     * 每一部分向左的位移
     */
    private static final long MACHINE_LEFT = SEQUENCE_BIT;
    private static final long TIMESTAMP_LEFT = SEQUENCE_BIT + MACHINE_BIT;
    /**
     * 机器码环境变量key
     */
    private static final String MACHINE_ID_ENV = "MACHINE_ID";
    private long machineId;
    private long sequence = 0L;
    private long lastStamp = -1L;

    /**
     * 获取环境变量值，设定机器ID，需要给容器设置不同的环境变量值
     */
    private SnowFlake() {
        try {
            String number = System.getenv().getOrDefault(MACHINE_ID_ENV, "0");
            this.init(Integer.parseInt(number));
        } catch (Exception e) {
            this.machineId = 0L;
        }
    }

    public static SnowFlake getInstance() {
        return SingletonHolder.INSTANCE;
    }

    /**
     * 解析ID
     *
     * @param id 雪花ID
     * @return ID构成
     */
    public static String decode(Long id) {
        long sequence = id & MAX_SEQUENCE;
        long machineId = id >> SEQUENCE_BIT & MAX_MACHINE_NUM;
        LocalDateTime dateTime = new Date((id >> TIMESTAMP_LEFT) + START_STAMP).toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
        return Arrays.asList(dateTime, machineId, sequence).toString();
    }

    /**
     * 初始化设备ID
     *
     * @param machineId 设备ID
     */
    public void init(long machineId) {
        if (machineId > MAX_MACHINE_NUM || machineId < 0) {
            throw new IllegalArgumentException("machineId can't be greater than MAX_MACHINE_NUM or less than 0");
        }
        this.machineId = machineId;
    }

    /**
     * 下一个ID
     *
     * @return ID
     */
    public synchronized long newId() {
        long currStamp = getNewStamp();
        if (currStamp < lastStamp) {
            throw new CommonException("Clock moved backwards. Refusing to generate id");
        }
        if (currStamp == lastStamp) {
            // 相同毫秒内，序列号自增
            sequence = (sequence + 1) & MAX_SEQUENCE;
            // 同一毫秒的序列数已经达到最大
            if (sequence == 0L) {
                currStamp = getNextMill();
            }
        } else {
            // 不同毫秒内，序列号置为0
            sequence = 0L;
        }
        lastStamp = currStamp;
        return (currStamp - START_STAMP) << TIMESTAMP_LEFT | machineId << MACHINE_LEFT | sequence;
    }

    private long getNextMill() {
        long mill = getNewStamp();
        while (mill <= lastStamp) {
            mill = getNewStamp();
        }
        return mill;
    }

    /**
     * return LocalDateTime.of(2025, 1, 1, 0, 0, 0).toInstant(ZoneOffset.of("+8")).toEpochMilli();
     * return LocalDateTime.of(2055, 12, 22, 23, 28, 35).toInstant(ZoneOffset.of("+8")).toEpochMilli();
     */
    private long getNewStamp() {
        return System.currentTimeMillis();
    }

    private static class SingletonHolder {
        private static final SnowFlake INSTANCE = new SnowFlake();
    }
}
