package com.digiwin.dap.middleware.commons.crypto;

import cn.hutool.core.util.HexUtil;
import cn.hutool.crypto.SecureUtil;
import com.digiwin.dap.middleware.exception.CommonException;
import com.digiwin.dap.middleware.commons.core.codec.Base64;
import com.digiwin.dap.middleware.commons.crypto.constant.AlgorithmEnum;
import com.digiwin.dap.middleware.commons.crypto.constant.TransformationEnum;
import com.digiwin.dap.middleware.commons.util.StrUtils;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.security.SecureRandom;

/**
 * 对称加密算法<br>
 * 在对称加密算法中，数据发信方将明文（原始数据）和加密密钥一起经过特殊加密算法处理后，使其变成复杂的加密密文发送出去。<br>
 * 收信方收到密文后，若想解读原文，则需要使用加密用过的密钥及相同算法的逆算法对密文进行解密，才能使其恢复成可读明文。<br>
 * 在对称加密算法中，使用的密钥只有一个，发收信双方都使用这个密钥对数据进行加密和解密，这就要求解密方事先必须知道加密密钥。<br>
 *
 * <br>
 * AES加密算法实现<br>
 * 高级加密标准（英语：Advanced Encryption Standard，缩写：AES），在密码学中又称Rijndael加密法<br>
 * 对于Java中AES的默认模式是：AES/ECB/PKCS5Padding，如果使用CryptoJS，请调整为：padding: CryptoJS.pad.Pkcs7
 *
 * <br>
 * <br>
 * 相关概念说明：
 * <pre>
 * mode:    加密算法模式，是用来描述加密算法（此处特指分组密码，不包括流密码，）在加密时对明文分组的模式，它代表了不同的分组方式
 * padding: 补码方式是在分组密码中，当明文长度不是分组长度的整数倍时，需要在最后一个分组中填充一些数据使其凑满一个分组的长度。
 * iv:      在对明文分组加密时，会将明文分组与前一个密文分组进行XOR运算（即异或运算），但是加密第一个明文分组时不存在“前一个密文分组”，
 *          因此需要事先准备一个与分组长度相等的比特序列来代替，这个比特序列就是偏移量。
 * </pre>
 *
 * <br>
 * <a href="https://blog.csdn.net/OrangeJack/article/details/82913804">相关概念</a>
 *
 * @author fobgochod
 * @since 1.0.0
 */
public class AES {

    private static final int KEY_LEN = 16;
    private static final int IV_SIZE = 16;
    private static final byte[] IV = StrUtils.bytes("ghUb#er57HBh(u%g");

    private AES() {
    }

    /**
     * 加密 默认的AES/ECB/PKCS5Padding  偏移向量null
     *
     * @param data 被加密的字符串
     * @param key  密钥
     * @return 加密后的Base64
     */
    public static String encrypt(String data, String key) {
        return encrypt(data, StrUtils.bytes(key));
    }

    /**
     * 解密 默认的AES/ECB/PKCS5Padding  偏移向量null
     *
     * @param data 被解密的Base64字符串
     * @param key  密钥
     * @return 解密后的String
     */
    public static String decrypt(String data, String key) {
        return decrypt(data, StrUtils.bytes(key));
    }

    /**
     * 加密 默认的AES/ECB/PKCS5Padding  偏移向量null
     *
     * @param data 被加密的字符串
     * @param key  密钥
     * @return 加密后的Base64
     */
    public static String encrypt(String data, byte[] key) {
        byte[] bytes = SecureUtil.aes(key).encrypt(data);
        return Base64.encode(bytes);
    }

    /**
     * 解密 默认的AES/ECB/PKCS5Padding  偏移向量null
     *
     * @param data 被解密的Base64字符串
     * @param key  密钥
     * @return 解密后的String
     */
    public static String decrypt(String data, byte[] key) {
        byte[] decode64 = Base64.decode(data);
        byte[] bytes = SecureUtil.aes(key).decrypt(decode64);
        return StrUtils.str(bytes);
    }

    /**
     * 加密 默认的AES/ECB/PKCS5Padding  偏移向量null
     *
     * @param data 被加密的字符串
     * @param key  密钥
     * @return 加密后的Hex
     */
    public static String encryptHex(String data, String key) {
        byte[] finalKey = paddingKey(key);
        byte[] bytes = SecureUtil.aes(finalKey).encrypt(data);
        char[] encodeHex = HexUtil.encodeHex(bytes, false);
        return new String(encodeHex);
    }

    /**
     * 解密 默认的AES/ECB/PKCS5Padding  偏移向量null
     *
     * @param data 被解密的Hex字符串
     * @param key  密钥
     * @return 解密后的String
     */
    public static String decryptHex(String data, String key) {
        byte[] finalKey = paddingKey(key);
        byte[] decodeHex = HexUtil.decodeHex(data.toCharArray());
        byte[] bytes = SecureUtil.aes(finalKey).decrypt(decodeHex);
        return StrUtils.str(bytes);
    }

    /**
     * 加密 AES/CBC/PKCS5Padding 偏移向量{@link AES#IV}
     *
     * @param data 被加密的字符串
     * @param key  密钥
     * @return 加密后的Base64
     */
    public static String encryptCBC(String data, String key) {
        return encryptCBC(data, key, IV);
    }

    /**
     * 解密  AES/CBC/PKCS5Padding 偏移向量{@link AES#IV}
     *
     * @param data 被解密的Base64字符串
     * @param key  密钥
     * @return 解密后的String
     */
    public static String decryptCBC(String data, String key) {
        return decryptCBC(data, key, IV);
    }

    /**
     * 加密 AES/CBC/PKCS5Padding 偏移向量{@link AES#IV}
     *
     * @param data 被加密的字符串
     * @param key  密钥
     * @param iv   偏移量
     * @return 加密后的Base64
     */
    public static String encryptCBC(String data, String key, byte[] iv) {
        byte[] bytes = StrUtils.bytes(data);
        byte[] finalKey = paddingKey(key);
        return Base64.encode(encrypt(bytes, finalKey, TransformationEnum.AES_CBC_PKCS5Padding.getValue(), iv));
    }

    /**
     * 解密  AES/CBC/PKCS5Padding 偏移向量{@link AES#IV}
     *
     * @param data 被解密的Base64字符串
     * @param key  密钥
     * @param iv   偏移量
     * @return 解密后的String
     */
    public static String decryptCBC(String data, String key, byte[] iv) {
        byte[] bytes = Base64.decode(data);
        byte[] finalKey = paddingKey(key);
        return StrUtils.str(decrypt(bytes, finalKey, TransformationEnum.AES_CBC_PKCS5Padding.getValue(), iv));
    }

    /**
     * 加密
     * <p>
     * <code>
     * cn.hutool.crypto.symmetric.AES aes = new cn.hutool.crypto.symmetric.AES(Mode.CBC, Padding.PKCS5Padding, key, KeyUtils.IV);
     * return aes.encrypt(data);
     * </code>
     *
     * @param data           被加密的bytes
     * @param key            密钥
     * @param transformation 算法/工作模式/填充模式
     * @param iv             偏移量
     * @return 加密后的bytes
     */
    @SuppressWarnings("findsecbugs:STATIC_IV")
    public static byte[] encrypt(byte[] data, byte[] key, String transformation, byte[] iv) {
        try {
            SecretKeySpec spec = new SecretKeySpec(key, AlgorithmEnum.AES.getValue());
            Cipher cipher = Cipher.getInstance(transformation);
            if (iv != null) {
                IvParameterSpec params = new IvParameterSpec(iv);
                cipher.init(Cipher.ENCRYPT_MODE, spec, params);
            } else {
                cipher.init(Cipher.ENCRYPT_MODE, spec);
            }
            return cipher.doFinal(data);
        } catch (Exception e) {
            throw new CommonException("AES加密异常", e);
        }
    }

    /**
     * 解密
     * <p>
     * <code>
     * cn.hutool.crypto.symmetric.AES aes = cn.hutool.crypto.symmetric.new AES(Mode.CBC, Padding.PKCS5Padding, key, KeyUtils.IV);
     * <br>
     * return aes.decrypt(data);
     * </code>
     *
     * @param data           被解密的bytes
     * @param key            密钥
     * @param transformation 算法/工作模式/填充模式
     * @param iv             偏移量
     * @return 解密后的bytes
     */
    public static byte[] decrypt(byte[] data, byte[] key, String transformation, byte[] iv) {
        try {
            SecretKeySpec spec = new SecretKeySpec(key, AlgorithmEnum.AES.getValue());
            Cipher cipher = Cipher.getInstance(transformation);
            if (iv != null) {
                IvParameterSpec params = new IvParameterSpec(iv);
                cipher.init(Cipher.DECRYPT_MODE, spec, params);
            } else {
                cipher.init(Cipher.DECRYPT_MODE, spec);
            }
            return cipher.doFinal(data);
        } catch (Exception e) {
            throw new CommonException("AES解密异常", e);
        }
    }

    /**
     * 加密
     *
     * @param data 被加密的字符串
     * @param key  密钥，不足16位自动补齐
     * @return 加密后的Base64
     */
    public static String encryptIvCBC(String data, String key) {
        byte[] bytes = StrUtils.bytes(data);
        byte[] finalKey = paddingKey(key);
        return Base64.encode(encryptIvCBC(bytes, finalKey));
    }

    /**
     * 解密
     *
     * @param data 被解密的Base64字符串
     * @param key  密钥，不足16位自动补齐
     * @return 解密后的String
     */
    public static String decryptIvCBC(String data, String key) {
        byte[] bytes = Base64.decode(data);
        byte[] finalKey = paddingKey(key);
        return StrUtils.str(decryptIvCBC(bytes, finalKey));
    }

    /**
     * 加密
     *
     * @param data 被加密的bytes
     * @param key  密钥
     * @return 加密后的bytes
     */
    public static byte[] encryptIvCBC(byte[] data, byte[] key) {
        return encryptIv(data, key, TransformationEnum.AES_CBC_PKCS5Padding.getValue());
    }

    /**
     * 解密
     *
     * @param data 被解密的bytes
     * @param key  密钥
     * @return 解密后的bytes
     */
    public static byte[] decryptIvCBC(byte[] data, byte[] key) {
        return decryptIv(data, key, TransformationEnum.AES_CBC_PKCS5Padding.getValue());
    }


    /**
     * 加密
     *
     * @param data           被加密的bytes
     * @param key            密钥
     * @param transformation 算法/工作模式/填充模式
     * @return IV+加密后的bytes
     */
    private static byte[] encryptIv(byte[] data, byte[] key, String transformation) {
        try {
            byte[] iv = new byte[IV_SIZE];
            SecureRandom random = new SecureRandom();
            random.nextBytes(iv);
            IvParameterSpec params = new IvParameterSpec(iv);

            SecretKeySpec secretKeySpec = new SecretKeySpec(key, AlgorithmEnum.AES.getValue());
            Cipher cipher = Cipher.getInstance(transformation);
            cipher.init(Cipher.ENCRYPT_MODE, secretKeySpec, params);

            byte[] encrypted = cipher.doFinal(data);
            byte[] combined = new byte[iv.length + encrypted.length];

            System.arraycopy(iv, 0, combined, 0, iv.length);
            System.arraycopy(encrypted, 0, combined, iv.length, encrypted.length);
            return combined;
        } catch (Exception e) {
            throw new CommonException("AES加密异常", e);
        }
    }

    /**
     * 解密
     *
     * @param data           IV+被解密的bytes
     * @param key            密钥
     * @param transformation 算法/工作模式/填充模式
     * @return 解密后的bytes
     */
    private static byte[] decryptIv(byte[] data, byte[] key, String transformation) {
        try {
            IvParameterSpec params = new IvParameterSpec(data, 0, IV_SIZE);
            SecretKeySpec secretKeySpec = new SecretKeySpec(key, AlgorithmEnum.AES.getValue());
            Cipher cipher = Cipher.getInstance(transformation);
            cipher.init(Cipher.DECRYPT_MODE, secretKeySpec, params);
            return cipher.doFinal(data, IV_SIZE, data.length - IV_SIZE);
        } catch (Exception e) {
            throw new CommonException("AES解密异常", e);
        }
    }

    /**
     * AES密钥最少128bits(16位)
     * <pre>
     * 1、不足16位，会补齐，密钥尽量使用16位，防止补齐
     * 2、超过16位会发生与前面异或超过，可能不兼容前端，不要超过16
     * </pre>
     *
     * @param key AES密钥
     * @return 16位的密钥
     */
    public static byte[] paddingKey(String key) {
        if (key == null) {
            throw new CommonException("AES key不能为空");
        }
        if (key.length() == KEY_LEN) {
            return StrUtils.bytes(key);
        }
        byte[] finalKey = new byte[KEY_LEN];
        int i = 0;
        for (byte b : StrUtils.bytes(key)) {
            finalKey[i++ % KEY_LEN] ^= b;
        }
        return finalKey;
    }

    /**
     * 获取AES密钥
     *
     * @param keySize 密钥模（modulus）长度（单位bit）
     * @return Base64编码aesKey字符串
     */
    public static byte[] generateKey(int keySize) {
        try {
            KeyGenerator keyGenerator = KeyGenerator.getInstance(AlgorithmEnum.AES.getValue());
            keyGenerator.init(keySize);
            return keyGenerator.generateKey().getEncoded();
        } catch (Exception e) {
            throw new CommonException("获取AES密钥异常,密钥长度:" + keySize, e);
        }
    }
}
