package com.digiwin.dap.middleware.util;

import com.digiwin.dap.middleware.exception.CommonException;
import com.digiwin.dap.middleware.serializer.*;
import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.json.JsonReadFeature;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;

import java.io.IOException;
import java.io.InputStream;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.util.Collections;
import java.util.Map;

/**
 * json相关的帮助类
 *
 * @author fobgochod
 * @since 1.0.0
 */
public final class JsonUtils {

    private static volatile ObjectMapper objectMapper = null;
    private static volatile ObjectMapper dmcObjectMapper = null;
    private static volatile ObjectMapper redisObjectMapper = null;

    static {
        createObjectMapper();
    }

    private JsonUtils() {
    }

    public static <T> Map<String, String> objToMap(T obj) {
        try {
            String json = objectMapper.writeValueAsString(obj);
            return objectMapper.readValue(json, new TypeReference<Map<String, String>>() {
            });
        } catch (Exception e) {
            throw new CommonException(e.getMessage(), e);
        }
    }

    public static <T> Map<String, Object> objToMapObject(T obj) {
        try {
            String json = objectMapper.writeValueAsString(obj);
            return objectMapper.readValue(json, new TypeReference<Map<String, Object>>() {
            });
        } catch (Exception e) {
            throw new CommonException(e.getMessage(), e);
        }
    }

    public static <T> Map<String, String> writeToMap(T obj) {
        try {
            String json = objectMapper.writeValueAsString(obj);
            return objectMapper.readValue(json, new TypeReference<Map<String, String>>() {
            });
        } catch (Exception ignored) {
            // Exception ignored intentionally
        }
        return Collections.emptyMap();
    }

    public static <T> Map<String, Object> writeToMapObject(T obj) {
        try {
            String json = objectMapper.writeValueAsString(obj);
            return objectMapper.readValue(json, new TypeReference<Map<String, Object>>() {
            });
        } catch (Exception ignored) {
            // Exception ignored intentionally
        }
        return Collections.emptyMap();
    }


    public static <T> String objToJson(T obj) {
        String result;
        try {
            result = objectMapper.writeValueAsString(obj);
        } catch (Exception e) {
            throw new CommonException(e.getMessage(), e);
        }
        return result;
    }

    public static <T> String writeValue(T obj) {
        try {
            return objectMapper.writeValueAsString(obj);
        } catch (Exception ignored) {
            // Exception ignored intentionally
        }
        return null;
    }

    public static <T> T jsonToObj(String json, Class<T> classType) {
        T t;
        try {
            t = objectMapper.readValue(json, classType);
        } catch (Exception e) {
            throw new CommonException(e.getMessage(), e);
        }
        return t;
    }

    public static <T> T jsonToObj(String json, TypeReference<T> valueTypeRef) {
        T t;
        try {
            t = objectMapper.readValue(json, valueTypeRef);
        } catch (Exception e) {
            throw new CommonException(e.getMessage(), e);
        }
        return t;
    }

    public static <T> T jsonToObj(InputStream json, Class<T> classType) {
        T t;
        try {
            t = objectMapper.readValue(json, classType);
        } catch (Exception e) {
            throw new CommonException(e.getMessage(), e);
        }
        return t;
    }

    public static <T> T jsonToObj(InputStream json, TypeReference<T> valueTypeRef) {
        T t;
        try {
            t = objectMapper.readValue(json, valueTypeRef);
        } catch (Exception e) {
            throw new CommonException(e.getMessage(), e);
        }
        return t;
    }

    public static <T> T readValue(String json, Class<T> classType) {
        try {
            return objectMapper.readValue(json, classType);
        } catch (IOException ignored) {
            // Exception ignored intentionally
        }
        return null;
    }

    public static <T> T readValue(String json, TypeReference<T> valueTypeRef) {
        try {
            return objectMapper.readValue(json, valueTypeRef);
        } catch (IOException ignored) {
            // Exception ignored intentionally
        }
        return null;
    }

    public static <T> T readValue(InputStream json, Class<T> valueType) {
        try {
            return objectMapper.readValue(json, valueType);
        } catch (IOException ignored) {
            // Exception ignored intentionally
        }
        return null;
    }

    public static <T> T readValue(InputStream json, TypeReference<T> valueTypeRef) {
        try {
            return objectMapper.readValue(json, valueTypeRef);
        } catch (IOException ignored) {
            // Exception ignored intentionally
        }
        return null;
    }

    public static ObjectMapper createObjectMapper() {
        if (objectMapper == null) {
            synchronized (JsonUtils.class) {
                if (objectMapper == null) {
                    objectMapper = getObjectMapper(1);
                }
            }
        }
        return objectMapper;
    }

    /**
     * DMC反序列化默认返回时间戳
     *
     * @return dmcObjectMapper
     */
    public static ObjectMapper createDmcObjectMapper() {
        if (dmcObjectMapper == null) {
            synchronized (JsonUtils.class) {
                if (dmcObjectMapper == null) {
                    dmcObjectMapper = getObjectMapper(2);
                }
            }
        }
        return dmcObjectMapper;
    }

    public static ObjectMapper createRedisObjectMapper() {
        if (redisObjectMapper == null) {
            synchronized (JsonUtils.class) {
                if (redisObjectMapper == null) {
                    redisObjectMapper = getObjectMapper(3);
                }
            }
        }
        return redisObjectMapper;
    }

    private static ObjectMapper getObjectMapper(int type) {
        ObjectMapper objectMapper = new ObjectMapper();
        objectMapper.registerModule(getJavaTimeModule());
        objectMapper.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, Boolean.FALSE);
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, Boolean.FALSE);
        // 属性为NULL 不序列化
        objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        // 允许出现特殊字符和转义符
        objectMapper.enable(JsonReadFeature.ALLOW_UNESCAPED_CONTROL_CHARS.mappedFeature());
        // 允许出现单引号
        objectMapper.configure(JsonParser.Feature.ALLOW_SINGLE_QUOTES, true);
        if (type == 1) {
            objectMapper.setDateFormat(new SimpleDateFormat(Constants.DATETIME_PATTERN));
        } else if (type == 3) {
            objectMapper.setDateFormat(new SimpleDateFormat(Constants.DATETIME_PATTERN));
            objectMapper.setVisibility(PropertyAccessor.ALL, JsonAutoDetect.Visibility.ANY);
            // 启用反序列化所需的類型信息,在屬性中添加@class
            objectMapper.activateDefaultTyping(objectMapper.getPolymorphicTypeValidator(), ObjectMapper.DefaultTyping.NON_FINAL, JsonTypeInfo.As.PROPERTY);
        }
        return objectMapper;
    }

    private static JavaTimeModule getJavaTimeModule() {
        JavaTimeModule javaTimeModule = new JavaTimeModule();
        javaTimeModule.addSerializer(LocalDate.class, new LocalDateSerializer());
        javaTimeModule.addDeserializer(LocalDate.class, new LocalDateDeserializer());
        javaTimeModule.addSerializer(LocalTime.class, new LocalTimeSerializer());
        javaTimeModule.addDeserializer(LocalTime.class, new LocalTimeDeserializer());
        javaTimeModule.addSerializer(LocalDateTime.class, new LocalDateTimeSerializer());
        javaTimeModule.addDeserializer(LocalDateTime.class, new LocalDateTimeDeserializer());
        javaTimeModule.addSerializer(Timestamp.class, new TimestampSerializer());
        javaTimeModule.addDeserializer(Timestamp.class, new TimestampDeserializer());
        SerializerUtil.addSerializer(javaTimeModule);
        return javaTimeModule;
    }
}
