package com.digiwin.dap.middleware.commons.crypto;

import cn.hutool.core.util.HexUtil;
import cn.hutool.crypto.SecureUtil;
import com.digiwin.dap.middleware.commons.util.StrUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 一些业务组装，最好不要用
 *
 * @author fobgochod
 */
public class PwdUtils {

    private static final Logger logger = LoggerFactory.getLogger(PwdUtils.class);

    private PwdUtils() {
    }

    /**
     * RSA解密出AES密钥，AES密钥解密密码
     *
     * @param passwordHash           客户端公钥加密的密码
     * @param clientEncryptPublicKey 服务端公钥加密的客户端公用
     * @param privateKey             服务端私钥
     * @return 密码
     * @since 1.0.4
     */
    public static String getPassWord(String passwordHash, String clientEncryptPublicKey, String privateKey) {
        String clientPublicKey = RSA.decrypt(clientEncryptPublicKey, privateKey);
        return decryptPwd(passwordHash, clientPublicKey);
    }

    /**
     * 通过RSA公钥获取AES密钥，再加密内容
     *
     * @param data            被加密的字符串
     * @param clientPublicKey RSA公钥
     * @return 加密后的Base64
     */
    public static String encryptPwd(String data, String clientPublicKey) {
        String aesKey = getAesKey(clientPublicKey);
        return AES.encryptCBC(data, aesKey);
    }

    /**
     * 通过RSA公钥获取AES密钥，再解密内容
     *
     * @param data            被解密的Base64字符串
     * @param clientPublicKey RSA公钥
     * @return 解密后的String
     */
    public static String decryptPwd(String data, String clientPublicKey) {
        String aesKey = getAesKey(clientPublicKey);
        return AES.decryptCBC(data, aesKey);
    }

    /**
     * AES加密HEX
     *
     * @param data 被加密的字符串
     * @param key  密钥
     * @return 加密后的Hex 或者 原文
     */
    public static String encryptHex(String data, String key) {
        try {
            return StrUtils.isEmpty(data) ? data : AES.encryptHex(data, key);
        } catch (Exception e) {
            logger.error("AES加密失败[{}]", data, e);
            return data;
        }
    }

    /**
     * AES解密
     *
     * @param data 被解密信息
     * @param key  密钥
     * @return 明文
     */
    public static String decryptHex(String data, String key) {
        try {
            return StrUtils.isEmpty(data) ? data : AES.decryptHex(data, key);
        } catch (Exception e) {
            logger.error("AES解密失败[{}]", data, e);
            return data;
        }
    }

    /**
     * AES加密
     *
     * @param data 加密内容
     * @param key  密钥
     * @return 加密后的HEX
     * @since 1.0.4
     */
    public static String encodeHex(String data, String key) {
        byte[] aesKey = HexUtil.decodeHex(key);
        byte[] bytes = SecureUtil.aes(aesKey).encrypt(data);
        return HexUtil.encodeHexStr(bytes, false);
    }

    /**
     * AES解密
     *
     * @param data 被解密的HEX字符串
     * @param key  密钥
     * @return 解密后的String
     * @since 1.0.4
     */
    public static String decodeHex(String data, String key) {
        byte[] aesKey = HexUtil.decodeHex(key);
        byte[] bytes = SecureUtil.aes(aesKey).decrypt(data);
        return StrUtils.str(bytes);
    }

    /**
     * 通过RSA公钥截取AES密钥
     *
     * @param clientPublicKey 客户端公用
     * @return AES密钥
     */
    public static String getAesKey(String clientPublicKey) {
        String key = clientPublicKey.substring(45, 54);
        return key + "digiwin";
    }
}
