package com.digiwin.dap.middleware.domain;

import com.digiwin.dap.middleware.constant.enums.AreaEnum;
import com.digiwin.dap.middleware.constant.enums.CloudEnum;

import java.util.Optional;

import static com.digiwin.dap.middleware.constant.enums.AreaEnum.*;
import static com.digiwin.dap.middleware.constant.enums.CloudEnum.*;

/**
 * 部署云区 enum
 *
 * @author fobgochod
 * @since 1.0.10
 */
@SuppressWarnings("java:S115")
public enum DeployAreaEnum {
    /**
     * 研发云-本地
     */
    DevelopLocal(Develop, local, ""),
    /**
     * 研发云-开发区
     */
    DevelopDev(Develop, dev, ""),
    /**
     *研发云- 地端环境
     */
    DevelopGround(Develop, ground, ""),
    /**
     * PaaS验证区
     */
    AliyunPaas(Aliyun, paas, "-paas.digiwincloud.com.cn"),
    /**
     * 阿里测试区
     */
    AliyunTest(Aliyun, test, "-test.digiwincloud.com.cn"),
    /**
     * 华为-压力测试区
     */
    HuaweiPressure(Huawei, pressure, "-pressure.digiwincloud.com.cn"),
    /**
     * 阿里云-自动化测试区
     */
    AliyunAutotest(Aliyun, autotest, "-autotest.digiwincloud.com.cn"),
    /**
     * 微软测试区
     */
    AzureTest(Azure, test, "-test.digiwincloud.com"),
    /**
     * 华为测试区
     */
    HuaweiTest(Huawei, test, "-hw-test.digiwincloud.com"),
    /**
     * 阿里正式区
     */
    AliyunProd(Aliyun, prod, ".digiwincloud.com.cn"),
    /**
     * 微软正式区
     */
    AzureProd(Azure, prod, ".digiwincloud.com"),
    /**
     * 华为正式区
     */
    HuaweiProd(Huawei, prod, "-hw.digiwincloud.com.cn");

    private final CloudEnum cloud;
    private final AreaEnum area;
    /**  URI后缀 */
    private final String suffix;

    DeployAreaEnum(CloudEnum cloud, AreaEnum area, String suffix) {
        this.cloud = cloud;
        this.area = area;
        this.suffix = suffix;
    }

    public static boolean isProd(String deployArea) {
        return Optional.ofNullable(get(deployArea)).map(x -> x.area.equals(prod)).orElse(false);
    }

    public static boolean isTest(String deployArea) {
        return Optional.ofNullable(get(deployArea)).map(x -> x.area.equals(test)).orElse(false);
    }

    /**
     * 是否阿里测试区、阿里正式区
     *
     * @param deployArea the deploy area
     * @return the boolean
     */
    public static boolean isAliyun(String deployArea) {
        return DeployAreaEnum.AliyunProd.name().equalsIgnoreCase(deployArea)
                || DeployAreaEnum.AliyunTest.name().equalsIgnoreCase(deployArea);
    }

    public static boolean isAzure(String deployArea) {
        return Optional.ofNullable(get(deployArea)).map(x -> x.cloud.equals(Azure)).orElse(false);
    }

    public static boolean isHuawei(String deployArea) {
        return Optional.ofNullable(get(deployArea)).map(x -> x.cloud.equals(Huawei)).orElse(false);
    }

    /**
     * 是否研发云、阿里PaaS
     *
     * @param deployArea the deployment area
     * @return the boolean
     */
    public static boolean isDev(String deployArea) {
        return DeployAreaEnum.AliyunPaas.name().equals(deployArea)
                || DeployAreaEnum.DevelopDev.name().equals(deployArea);
    }

    /**
     * 获取部署云区
     *
     * @param cloud the cloud
     * @param area  the area
     * @return the deployment area enum
     */
    public static DeployAreaEnum get(String cloud, String area) {
        for (DeployAreaEnum anEnum : DeployAreaEnum.values()) {
            if (anEnum.cloud.name().equalsIgnoreCase(cloud) && anEnum.area.name().equalsIgnoreCase(area)) {
                return anEnum;
            }
        }
        return null;
    }

    /**
     * 获取部署云区
     *
     * @param deployArea the deployment area
     * @return the deployment area enum
     */
    public static DeployAreaEnum get(String deployArea) {
        for (DeployAreaEnum anEnum : DeployAreaEnum.values()) {
            if (anEnum.name().equalsIgnoreCase(deployArea)) {
                return anEnum;
            }
        }
        return null;
    }

    public CloudEnum getCloud() {
        return cloud;
    }

    public AreaEnum getArea() {
        return area;
    }

    public String getSuffix() {
        return suffix;
    }

    public String getCloudName() {
        return cloud.name();
    }

    public String getAreaName() {
        return area.name();
    }
}
