package com.digiwin.dap.middleware.commons.crypto;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.ListUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.RandomUtil;
import cn.hutool.crypto.SecureUtil;
import com.digiwin.dap.middleware.serializer.Constants;
import com.digiwin.dap.middleware.util.JsonUtils;

import java.lang.reflect.Array;
import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * 签名工具类
 *
 * @author fobgochod
 * @since 1.0.0
 */
public class SignUtils {

    private static final String SIGN = "sign";
    private static final String AND = "&";
    private static final String EQUAL_SIGN = "=";

    /**
     * 签名
     *
     * @param key  签名密钥
     * @param data 签名参数
     * @return 签名结果
     */
    public static String sign(String key, String data) {
        return SecureUtil.hmacSha256(key).digestBase64(data, StandardCharsets.UTF_8, true);
    }

    /**
     * 签名
     *
     * @param signParams  签名参数
     * @param key         签名密钥
     * @param otherParams 其他参数
     * @return 签名对象
     */
    @SafeVarargs
    public static String sign(Map<String, String> signParams, String key, Map<String, String>... otherParams) {
        // 1.初始化签名参数
        signParams.putIfAbsent("timestamp", LocalDateTime.now().format(Constants.PURE_DATETIME_FORMATTER));
        signParams.putIfAbsent("nonce", RandomUtil.randomString(16));
        String signParam = sortParam(signParams);

        // 2.分别排序可变参数
        String[] params = new String[otherParams.length + 1];
        params[0] = signParam;
        for (int i = 0; i < otherParams.length; i++) {
            params[i + 1] = sortParam(otherParams[i]);
        }

        // 3.组合所有类型的数据
        List<String> combineParams = ListUtil.toLinkedList(params);
        CollUtil.removeEmpty(combineParams);
        String signString = CollUtil.join(combineParams, AND);
        String sign = sign(key, signString);
        signParams.put(SIGN, sign);
        return sign;
    }

    /**
     * 验签
     *
     * @param signParams  签名参数
     * @param key         签名密钥
     * @param otherParams 其他参数
     * @return 是否成功
     */
    @SafeVarargs
    public static boolean verify(Map<String, String> signParams, String key, Map<String, String>... otherParams) {
        // 1.获取signString
        String oldSign = signParams.remove(SIGN);
        String signParam = sortParam(signParams);

        // 2.分别排序可变参数
        String[] params = new String[otherParams.length + 1];
        params[0] = signParam;
        for (int i = 0; i < otherParams.length; i++) {
            params[i + 1] = sortParam(otherParams[i]);
        }

        // 3.合并验证参数， 按照sign、query、body各自排序拼接
        List<String> combineParams = ListUtil.toLinkedList(params);
        CollUtil.removeEmpty(combineParams);
        String signString = CollUtil.join(combineParams, AND);

        // 4.验签
        String newSign = sign(key, signString);
        return Objects.equals(newSign, oldSign);
    }

    /**
     * 针对sign、query、body分别排序验签
     *
     * @param signParams  签名参数
     * @param key         签名密钥
     * @param queryParams query params
     * @param bodyParam   body params
     * @return 是否成功
     */
    public static boolean verify(Map<String, String> signParams, String key, Map<String, String> queryParams, String bodyParam) {
        // 1.获取signString
        String oldSign = signParams.remove(SIGN);
        String signParam = sortParam(signParams);

        // 2.获取queryString
        String queryParam = sortParam(queryParams);

        // 3.合并验证参数， 按照sign、query、body各自排序拼接
        List<String> combineParams = ListUtil.toLinkedList(signParam, queryParam, bodyParam);
        CollUtil.removeEmpty(combineParams);
        String signString = CollUtil.join(combineParams, AND);

        // 4.验签
        String newSign = sign(key, signString);
        return Objects.equals(newSign, oldSign);
    }

    public static String sortParam(String json) {
        Map<String, String> objectMap = JsonUtils.jsonToObj(json, Map.class);
        return sortParam(objectMap);
    }

    public static String sortParam(Object object) {
        Map<String, String> objectMap = JsonUtils.objToMap(object);
        return sortParam(objectMap);
    }

    public static String sortParam(Map<String, String> params) {
        return MapUtil.sortJoin(params, AND, EQUAL_SIGN, true);
    }

    @SafeVarargs
    private static <T> T[] prepend(T element, T... elements) {
        @SuppressWarnings("unchecked")
        T[] result = (T[]) Array.newInstance(element.getClass(), 1 + elements.length);
        result[0] = element;
        System.arraycopy(elements, 0, result, 1, elements.length);
        return result;
    }
}
