package com.digiwin.dap.middleware.util;

import com.auth0.jwt.JWT;
import com.auth0.jwt.JWTVerifier;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.exceptions.JWTVerificationException;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.digiwin.dap.middleware.commons.util.StrUtils;
import com.digiwin.dap.middleware.commons.crypto.constant.KeyConstant;

import java.util.Date;

/**
 * jwt帮助类型
 *
 * @author chenxsa
 */
public final class JwtUtils {

    private static final long EXPIRE_TIME = 24 * 60 * 60 * 1000L;
    private static final String KEY = KeyConstant.OTHER;
    private static final String SUB = "iam";
    private static final String ISSUER = "digiwin.dap.middleware.iam.";

    /**
     * 校验token是否正确
     *
     * @param token 密钥
     * @return 是否正确
     */
    public static boolean verify(String token) {
        return verify(token, "");
    }

    /**
     * 校验token是否正确
     *
     * @param token  密钥
     * @param issuer 签名用户
     * @return 是否正确
     */
    public static boolean verify(String token, String issuer) {
        try {
            Algorithm algorithm = Algorithm.HMAC256(KEY);
            JWTVerifier verifier =
                    JWT.require(algorithm)
                            .withIssuer(ISSUER + issuer)
                            .withSubject(SUB)
                            .build();
            verifier.verify(token);
            return true;
        } catch (JWTVerificationException exception) {
            return false;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 获得token中的信息
     *
     * @param token  jwt数据
     * @param issuer 签名用户
     * @return token中包含的用户名
     */
    public static String getData(String token, String issuer) {
        Algorithm algorithm = Algorithm.HMAC256(KEY);
        JWTVerifier verifier =
                JWT.require(algorithm)
                        .withIssuer(ISSUER + issuer)
                        .withSubject(SUB)
                        .build();
        DecodedJWT jwt = verifier.verify(token);
        return jwt.getClaim("data").asString();
    }

    /**
     * 获得token中的信息无需secret解密也能获得
     *
     * @param token  jwt数据
     * @param issuer 签名用户
     * @return token中包含的用户名
     */
    public static String getDataWithoutVerify(String token, String issuer) {
        DecodedJWT jwt = JWT.decode(token);
        return jwt.getClaim("data").asString();
    }


    public static String sign(String json, String key, String issuer, long expire) {
        if (StrUtils.isEmpty(json)) {
            throw new IllegalArgumentException("json 不能为null");
        }
        try {
            Date date = new Date(System.currentTimeMillis() + expire);
            Algorithm algorithm = Algorithm.HMAC256(key);
            // 附带username信息
            return JWT.create()
                    .withIssuer(ISSUER + issuer)
                    .withSubject(SUB)
                    .withClaim("data", json)
                    .withExpiresAt(date)
                    .sign(algorithm);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }


    public static String sign(String json, String key, String issuer) {
        if (StrUtils.isEmpty(json)) {
            throw new IllegalArgumentException("json 不能为null");
        }
        try {
            Algorithm algorithm = Algorithm.HMAC256(key);
            // 附带username信息
            return JWT.create()
                    .withIssuer(ISSUER + issuer)
                    .withSubject(SUB)
                    .withClaim("data", json)
                    .sign(algorithm);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }


    public static String sign(String json, String issuer, long expire) {
        return sign(json, KEY, issuer, expire);
    }


    public static String sign(String json, String issuer, boolean isExpire) {
        if (isExpire) {
            return sign(json, KEY, issuer, EXPIRE_TIME);
        } else {
            return sign(json, KEY, issuer);
        }
    }

    /**
     * 生成签名,30min后过期
     *
     * @param json json
     * @return 加密的token
     */
    public static String sign(String json) {
        return sign(json, KEY, "", EXPIRE_TIME);
    }
}
