package com.digiwin.dap.middleware.commons.crypto;

import cn.hutool.crypto.digest.DigestAlgorithm;
import cn.hutool.crypto.digest.DigestUtil;
import com.auth0.jwt.JWT;
import com.auth0.jwt.algorithms.Algorithm;
import com.digiwin.dap.middleware.commons.core.codec.Base64;
import com.digiwin.dap.middleware.commons.util.StrUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.security.MessageDigest;

/**
 * 摘要算法工具类
 *
 * @author fobgochod
 * @since 1.0.0
 */
public class DigestUtils {

    private static final Logger logger = LoggerFactory.getLogger(DigestUtils.class);

    private DigestUtils() {
    }

    /**
     * 计算sha256摘要值
     *
     * @param data 被摘要数据
     * @return SHA-256摘要
     */
    public static String sha256(String data) {
        return sha256(data, 2);
    }

    /**
     * 计算sha256摘要值
     *
     * @param data  被摘要数据
     * @param count 摘要计算次数
     * @return SHA-256摘要
     */
    public static String sha256(String data, int count) {
        byte[] digest = StrUtils.bytes(data);
        for (int i = 0; i < count; i++) {
            digest = DigestUtil.sha256(digest);
        }
        return Base64.encode(digest);
    }

    /**
     * 计算32位MD5摘要值，并转为16进制字符串
     *
     * @param data 被摘要数据
     * @return md5摘要
     */
    public static String md5Hex(String data) {
        return DigestUtil.md5Hex(StrUtils.bytes(data));
    }

    /**
     * 计算MD5摘要值
     *
     * @param data 被摘要数据
     * @param salt 盐值
     * @return md5摘要
     */
    public static String md5(String data, String salt) {
        try {
            MessageDigest digest = MessageDigest.getInstance(DigestAlgorithm.MD5.getValue());
            digest.update(Base64.decode(salt));
            byte[] bytes = digest.digest(data.getBytes());
            StringBuilder sb = new StringBuilder();
            for (byte aByte : bytes) {
                sb.append(String.format("%02x", aByte));
            }
            return sb.toString();
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
        return null;
    }

    /**
     * 计算MD5摘要值
     *
     * @param data 被加密值
     * @return Base64编译摘要值
     * @since 1.0.5
     */
    public static String md5(byte[] data) {
        try {
            MessageDigest md = MessageDigest.getInstance(DigestAlgorithm.MD5.getValue());
            md.reset();
            md.update(data);
            byte[] digest = md.digest();
            return Base64.encode(digest);
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
        return null;
    }

    /**
     * 根据应用id去创建token信息
     * 保证每一区的token一致
     *
     * @param sysId 应用ID
     * @param key   密钥
     * @return appToken
     * @since 1.0.4
     */
    public static String getAppToken(String sysId, String key) {
        Algorithm algorithm = Algorithm.HMAC256(key);
        return JWT.create()
                .withClaim("id", sysId)
                .withClaim("sid", 0L)
                .sign(algorithm);
    }
}
