package com.digiwin.dap.middleware.commons.util;

import com.digiwin.dap.middleware.commons.core.codec.URL;
import com.digiwin.dap.middleware.commons.crypto.AES;
import com.digiwin.dap.middleware.commons.crypto.SignUtils;
import com.digiwin.dap.middleware.domain.DapEncrypt;
import com.digiwin.dap.middleware.domain.DapSign;
import com.digiwin.dap.middleware.util.JsonUtils;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * 加密工具类
 * 中间件接口加密请求参数加密处理
 *
 * @since 1.0.8
 */
public class EncryptUtils {

    private EncryptUtils() {
    }

    /**
     * 加密方法，将请求对象根据密钥进行加密处理，返回加密对象
     *
     * @param key  加密密钥
     * @param data 加密对象
     * @return 对象加密之后生成的加密字符串
     */
    public static String encrypt(String key, Object data) {
        String dataJson = JsonUtils.objToJson(data);
        return AES.encryptIvCBC(dataJson, key);
    }

    /**
     * 解密加密对象
     *
     * @param key        解密密钥
     * @param eData      加密对象
     * @param targetType 解密对象
     * @param <T>        对象类型
     * @return 加密字符串解密之后的对象
     */
    public static <T> T decrypt(String key, String eData, Class<T> targetType) {
        String data = AES.decryptIvCBC(eData, key);
        return JsonUtils.jsonToObj(data, targetType);
    }

    /**
     * 加签，参数不进行加密处理
     * <p>
     * digi-middleware-sign-arg
     *
     * @param key         加签密钥
     * @param requestBody 请求对象
     * @return 加签对象
     */
    public static String sign(String key, Object requestBody) {
        return sign(key, Collections.emptyMap(), requestBody);
    }

    /**
     * 加签，参数不进行加密处理
     *
     * @param key         加签密钥
     * @param queryParams 请求查询参数
     * @param requestBody 请求对象
     * @return 加签对象
     */
    public static String sign(String key, Map<String, String> queryParams, Object requestBody) {
        DapSign dapSign = signInfo(key, queryParams, requestBody);
        return URL.encode(JsonUtils.objToJson(dapSign));
    }

    /**
     * 加签，参数不进行加密处理
     *
     * @param key         加签密钥
     * @param requestBody 请求对象
     * @return 加签对象
     */
    public static DapSign signInfo(String key, Object requestBody) {
        return signInfo(key, Collections.emptyMap(), requestBody);
    }

    /**
     * 加签，参数不进行加密处理
     *
     * @param key         加签密钥
     * @param queryParams 请求查询参数
     * @param requestBody 请求对象
     * @return 加签对象
     */
    public static DapSign signInfo(String key, Map<String, String> queryParams, Object requestBody) {
        Map<String, String> signParams = new HashMap<>();
        Map<String, String> bodyParams = requestBody == null ? Collections.emptyMap() : JsonUtils.objToMap(requestBody);
        String signData = SignUtils.sign(signParams, key, queryParams, bodyParams);
        DapSign dapSign = new DapSign();
        dapSign.setTimestamp(signParams.get("timestamp"));
        dapSign.setNonce(signParams.get("nonce"));
        dapSign.setSign(signData);
        return dapSign;
    }

    /**
     * 加密加签，请求参数进行加密处理并且返回加签标记
     *
     * @param key         加签密钥
     * @param requestBody 请求对象
     * @return 加签对象
     */
    public static DapEncrypt encryptSign(String key, Object requestBody) {
        return encryptSign(key, Collections.emptyMap(), requestBody);
    }

    /**
     * 加密加签，请求参数进行加密处理并且返回加签标记
     *
     * @param key         加签密钥
     * @param queryParams 查询参数
     * @param requestBody 请求对象
     * @return 加签对象
     */
    public static DapEncrypt encryptSign(String key, Map<String, String> queryParams, Object requestBody) {
        String eData = encrypt(key, requestBody);
        DapEncrypt body = new DapEncrypt(eData);
        body.setSignHeader(sign(key, queryParams, body));
        return body;
    }
}
