package com.digiwin.dap.middleware.commons.crypto;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.collection.ListUtil;
import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.LocalDateTimeUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.CharsetUtil;
import cn.hutool.core.util.RandomUtil;
import cn.hutool.crypto.SecureUtil;
import com.digiwin.dap.middleware.commons.crypto.domain.DapEncryptDTO;
import com.digiwin.dap.middleware.commons.crypto.domain.DapSignInfo;
import com.digiwin.dap.middleware.serializer.Constants;
import com.digiwin.dap.middleware.util.JsonUtils;

import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;

/**
 * 签名工具类
 *
 * @author fobgochod
 * @since 1.0.0
 */
public class SignUtils {

    private static final String SIGN = "sign";
    private static final String AND = "&";
    private static final String EQUAL_SIGN = "=";

    private SignUtils() {
    }

    /**
     * 签名
     *
     * @param key  签名密钥
     * @param data 签名参数
     * @return 签名结果
     */
    public static String sign(String key, String data) {
        return SecureUtil.hmacSha256(key).digestBase64(data, StandardCharsets.UTF_8, true);
    }

    /**
     * 签名
     *
     * @param signParams  签名参数
     * @param key         签名密钥
     * @param otherParams 其他参数
     * @return 签名对象
     */
    @SafeVarargs
    public static String sign(Map<String, String> signParams, String key, Map<String, String>... otherParams) {
        // 1.初始化签名参数
        signParams.putIfAbsent("timestamp", LocalDateTime.now().format(Constants.PURE_DATETIME_FORMATTER));
        signParams.putIfAbsent("nonce", RandomUtil.randomString(16));
        String signParam = sortParam(signParams);

        // 2.分别排序可变参数
        String[] params = new String[otherParams.length + 1];
        params[0] = signParam;
        for (int i = 0; i < otherParams.length; i++) {
            params[i + 1] = sortParam(otherParams[i]);
        }

        // 3.组合所有类型的数据
        List<String> combineParams = ListUtil.toLinkedList(params);
        CollUtil.removeEmpty(combineParams);
        String signString = CollUtil.join(combineParams, AND);
        String sign = sign(key, signString);
        signParams.put(SIGN, sign);
        return sign;
    }

//    /**
//     * DAP加签
//     * @param signParams 签名对象
//     * @param key        签名密钥
//     * @param otherParams 其它参数
//     * @return            签名对象
//     */
//    public static DapSignInfo sign(Object signParams, String key, Map<String, String>... otherParams){
//        String eData=EncryptUtils.encrypt(key,signParams);
//        DapEncryptDTO encryptDTO=new DapEncryptDTO();
//        encryptDTO.seteData(eData);
//        return signCore(key,encryptDTO,otherParams);
//    }
//
//    /**
//     * DAP加签
//     * @param appSecret
//     * @param data
//     * @param otherParams
//     * @return
//     */
//    private static DapSignInfo signCore(String appSecret, DapEncryptDTO data, Map<String,String>... otherParams){
//        DapSignInfo signInfo = new DapSignInfo();
//        signInfo.setTimestamp(LocalDateTimeUtil.format(LocalDateTime.now(), DatePattern.PURE_DATETIME_FORMATTER));
//        signInfo.setNonce(RandomUtil.randomString(16));
//        // 1.sign header
//        String signHeaderParams  = sortParam(signInfo);
//        //2.query param
//        String requestParams=null;
//        if(null!=otherParams&& otherParams.length>0){
//            requestParams=sortParam(otherParams);
//        }
//        //3.query body
//        String bodyParams = sortParam(data);
//        //组合所有类型的数据
//        List<String> signList= ListUtil.toList(signHeaderParams,requestParams,bodyParams);
//        CollUtil.removeEmpty(signList);
//        signInfo.setSign(sign(appSecret,CollUtil.join(signList,"&")));
//        return signInfo;
//    }


    /**
     * 验签
     *
     * @param signParams  签名参数
     * @param key         签名密钥
     * @param otherParams 其他参数
     * @return 是否成功
     */
    @SafeVarargs
    public static boolean verify(Map<String, String> signParams, String key, Map<String, String>... otherParams) {
        // 1.获取signString
        String oldSign = signParams.remove(SIGN);
        String signParam = sortParam(signParams);

        // 2.分别排序可变参数
        String[] params = new String[otherParams.length + 1];
        params[0] = signParam;
        for (int i = 0; i < otherParams.length; i++) {
            params[i + 1] = sortParam(otherParams[i]);
        }

        // 3.合并验证参数， 按照sign、query、body各自排序拼接
        List<String> combineParams = ListUtil.toLinkedList(params);
        CollUtil.removeEmpty(combineParams);
        String signString = CollUtil.join(combineParams, AND);

        // 4.验签
        String newSign = sign(key, signString);
        return Objects.equals(newSign, oldSign);
    }

    /**
     * 针对sign、query、body分别排序验签
     *
     * @param signParams  签名参数
     * @param key         签名密钥
     * @param queryParams query params
     * @param bodyParam   body params
     * @return 是否成功
     */
    public static boolean verify(Map<String, String> signParams, String key, Map<String, String> queryParams, String bodyParam) {
        // 1.获取signString
        String oldSign = signParams.remove(SIGN);
        String signParam = sortParam(signParams);

        // 2.获取queryString
        String queryParam = sortParam(queryParams);

        // 3.合并验证参数， 按照sign、query、body各自排序拼接
        List<String> combineParams = ListUtil.toLinkedList(signParam, queryParam, bodyParam);
        CollUtil.removeEmpty(combineParams);
        String signString = CollUtil.join(combineParams, AND);

        // 4.验签
        String newSign = sign(key, signString);
        return Objects.equals(newSign, oldSign);
    }

    public static String sortParam(String json) {
        Map<String, String> objectMap = JsonUtils.jsonToObj(json, Map.class);
        return sortParam(objectMap);
    }

    public static String sortParam(Object object) {
        Map<String, String> objectMap = JsonUtils.objToMap(object);
        return sortParam(objectMap);
    }

    public static String sortParam(Map<String, String> params) {
        return MapUtil.sortJoin(params, AND, EQUAL_SIGN, true);
    }


}
