package com.digiwin.dap.middleware.util;

import com.digiwin.dap.middleware.exception.CommonException;
import com.fasterxml.jackson.annotation.JsonIgnore;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * 复制对象
 */
@SuppressWarnings("java:S3011")
public class ObjectCopyUtil {

    private ObjectCopyUtil() {
    }

    /**
     * 获取对象所有字段
     * <p>
     * 排除合成字段
     * 为了收集数据，JaCoCo会植入私有静态$jacocoData变量和私有静态方法$jacocoInit()，
     * 如果使用反射循环成员变量进而拼凑set或者get方法[get$jacocoData()]，会产生NoSuchMethodException，
     * 建议在循环中利用isSynthetic()方法检查成员变量
     * <p>
     * 排除static字段
     * 引入日志框架异常
     * private static final Logger logger
     *
     * @param object 对象
     * @return 字段
     */
    public static List<Field> getFields(Object object) {
        List<Field> fields = new ArrayList<>();
        Class<?> aClass = object.getClass();
        // 当父类为null的时候说明到达了最上层的父类(Object类).
        while (aClass != null) {
            Field[] declaredFields = aClass.getDeclaredFields();
            Arrays.stream(declaredFields).filter(field -> !field.isSynthetic() && !Modifier.isStatic(field.getModifiers())).forEach(fields::add);
            // 得到父类,然后赋给自己
            aClass = aClass.getSuperclass();
        }
        return fields;
    }

    /**
     * 将origin属性注入到destination中
     *
     * @param origin      源对象
     * @param destination 目标对象
     */
    public static void mergeObject(Object origin, Object destination) {

        if (origin == null || destination == null) {
            return;
        }
        if (!origin.getClass().equals(destination.getClass())) {
            return;
        }
        List<Field> fieldList = getFields(origin);

        for (Field field : fieldList) {
            field.setAccessible(true);
            try {
                Object value = field.get(origin);
                if (null != value) {
                    field.set(destination, value);
                }
                field.setAccessible(false);
            } catch (IllegalAccessException e) {
                throw new CommonException("对象合并错误！", e);
            }
        }
    }

    /**
     * 将origin属性注入到destination中
     *
     * @param origin      源对象
     * @param destination 目标对象
     */
    public static void mergeDifferentTypeObject(Object origin, Object destination) {
        if (origin == null || destination == null) {
            return;
        }
        List<Field> originFieldList = getFields(origin);
        List<Field> destinationFieldList = getFields(destination);

        for (Field origField : originFieldList) {
            for (Field destField : destinationFieldList) {
                if (!origField.getName().equals(destField.getName())) {
                    continue;
                }
                if (origField.getAnnotation(JsonIgnore.class) == null) {
                    mergeField(origField, destField, origin, destination);
                }
            }
        }
    }

    private static void mergeField(Field origField, Field destField, Object origin, Object destination) {
        try {
            origField.setAccessible(true);
            Object value = origField.get(origin);
            origField.setAccessible(false);

            if (value != null) {
                destField.setAccessible(true);
                destField.set(destination, value);
                destField.setAccessible(false);
            }
        } catch (IllegalAccessException e) {
            throw new CommonException("对象复制错误", e);
        }
    }
}
