package com.digiwin.dap.middleware.util;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * 方法类
 *
 * @author chenxsa
 */
@SuppressWarnings("java:S3011")
public class ReflectionUtils {

    private static final Logger logger = LoggerFactory.getLogger(ReflectionUtils.class);

    private ReflectionUtils() {
    }

    /**
     * 循环向上转型, 获取对象的 DeclaredMethod
     *
     * @param object         : 子类对象
     * @param methodName     : 父类中的方法名
     * @param parameterTypes : 父类中的方法参数类型
     * @return 父类中的方法对象
     */

    public static Method getDeclaredMethod(Object object, String methodName, Class<?>... parameterTypes) {
        Method method = null;

        for (Class<?> clazz = object.getClass(); clazz != Object.class; clazz = clazz.getSuperclass()) {
            try {
                method = clazz.getDeclaredMethod(methodName, parameterTypes);
                return method;
            } catch (NoSuchMethodException | SecurityException e) {
                // 这里甚么都不要做！并且这里的异常必须这样写，不能抛出去。
                // 如果这里的异常打印或者往外抛，则就不会执行clazz = clazz.getSuperclass(),最后就不会进入到父类中了
                logger.error(e.getMessage(), e);
            }
        }

        return null;
    }

    /**
     * 直接调用对象方法, 而忽略修饰符(private, protected, default)
     *
     * @param object         : 子类对象
     * @param methodName     : 父类中的方法名
     * @param parameterTypes : 父类中的方法参数类型
     * @param parameters     : 父类中的方法参数
     * @return 父类中方法的执行结果
     */

    public static Object invokeMethod(Object object, String methodName, Class<?>[] parameterTypes,
                                      Object[] parameters) {
        // 根据 对象、方法名和对应的方法参数 通过反射 调用上面的方法获取 Method 对象
        Method method = getDeclaredMethod(object, methodName, parameterTypes);
        try {
            if (method != null) {
                // 抑制Java对方法进行检查,主要是针对私有方法而言
                method.setAccessible(true);  // WARNING: Reflective access to private fields
                // 调用object 的 method 所代表的方法，其方法的参数是 parameters
                return method.invoke(object, parameters);
            }
        } catch (IllegalArgumentException | IllegalAccessException | InvocationTargetException e) {
            logger.error(e.getMessage(), e);
        }
        return null;
    }

    /**
     * 循环向上转型, 获取对象上所有字段
     *
     * @param clazz :  类型
     * @return 字段
     */
    public static List<Field> getDeclaredFields(Class<?> clazz) {
        List<Field> result = new ArrayList<>();

        Field[] fields = clazz.getDeclaredFields();
        Collections.addAll(result, fields);
        Class<?> superClass = clazz.getSuperclass();
        if (superClass == Object.class) {
            return result;
        }
        List<Field> superFields = getDeclaredFields(superClass);
        result.addAll(superFields);
        return result;
    }

    /**
     * 循环向上转型, 获取对象的所有字段
     *
     * @param clazz     : 类型
     * @param fieldName : 属性名
     * @return 类中的所有属性对象
     */

    public static Field getDeclaredField(Class<?> clazz, String fieldName) {
        Field field;
        try {
            field = clazz.getDeclaredField(fieldName);
            return field;
        } catch (NoSuchFieldException e) {
            // 继续操作父类
        }

        Class<?> superClass = clazz.getSuperclass();
        if (superClass == Object.class) {
            return null;
        }
        try {
            field = getDeclaredField(superClass, fieldName);
            if (field != null) {
                return field;
            }
        } catch (Exception e) {
            // 这里甚么都不要做！并且这里的异常必须这样写，不能抛出去。
            // 如果这里的异常打印或者往外抛，则就不会执行clazz = clazz.getSuperclass(),最后就不会进入到父类中了
        }
        return null;
    }

    /**
     * 直接设置对象属性值, 忽略 private/protected 修饰符, 也不经过 setter
     *
     * @param object    : 子类对象
     * @param fieldName : 父类中的属性名
     * @param value     : 将要设置的值
     */

    public static void setFieldValue(Object object, String fieldName, Object value) {
        try {
            // 根据 对象和属性名通过反射 调用上面的方法获取 Field对象
            Field field = getDeclaredField(object.getClass(), fieldName);
            if (field != null) {
                // 抑制Java对其的检查
                field.setAccessible(true);  // WARNING: Reflective access to private fields
                // 将 object 中 field 所代表的值 设置为 value
                field.set(object, value);
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
    }

    /**
     * 直接读取对象的属性值, 忽略 private/protected 修饰符, 也不经过 getter
     *
     * @param object    : 子类对象
     * @param fieldName : 父类中的属性名
     * @return : 父类中的属性值
     */

    public static Object getFieldValue(Object object, String fieldName) {
        try {
            // 根据 对象和属性名通过反射 调用上面的方法获取 Field对象
            Field field = getDeclaredField(object.getClass(), fieldName);
            if (field != null) {
                // 抑制Java对其的检查
                field.setAccessible(true); // WARNING: Reflective access to private fields
                // 获取 object 中 field 所代表的属性值
                return field.get(object);
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
        return null;
    }
}