package com.digiwin.dap.middle.console.config;

import org.springframework.data.redis.connection.RedisPassword;
import org.springframework.data.redis.connection.RedisStandaloneConfiguration;
import org.springframework.data.redis.connection.jedis.JedisClientConfiguration;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.RedisSerializer;
import redis.clients.jedis.JedisPoolConfig;

import java.time.Duration;

public class ConsoleRedis {

    private static volatile RedisTemplate<String, Object> template;

    public static RedisTemplate<String, Object> get(ConsoleRedisProperties env) {
        if (template == null) {
            synchronized (ConsoleRedis.class) {
                if (template == null) {
                    RedisTemplate<String, Object> temp = new RedisTemplate<>();
                    temp.setConnectionFactory(createJedisConnectionFactory(env));
                    temp.setKeySerializer(RedisSerializer.string());
                    temp.setValueSerializer(RedisSerializer.java());
                    temp.afterPropertiesSet();
                    template = temp;
                }
            }
        }
        return template;
    }

    private static JedisConnectionFactory createJedisConnectionFactory(ConsoleRedisProperties env) {
        JedisClientConfiguration clientConfiguration = getJedisClientConfiguration();
        JedisConnectionFactory jedisConnectionFactory = new JedisConnectionFactory(getStandaloneConfig(env), clientConfiguration);
        jedisConnectionFactory.afterPropertiesSet();
        return jedisConnectionFactory;
    }

    private static JedisClientConfiguration getJedisClientConfiguration() {
        JedisClientConfiguration.JedisClientConfigurationBuilder builder = applyProperties(JedisClientConfiguration.builder());
        builder.usePooling().poolConfig(jedisPoolConfig());
        return builder.build();
    }

    private static RedisStandaloneConfiguration getStandaloneConfig(ConsoleRedisProperties env) {
        RedisStandaloneConfiguration config = new RedisStandaloneConfiguration();
        config.setHostName(env.getRedisHost());
        config.setPort(env.getRedisPort());
        config.setPassword(RedisPassword.of(env.getRedisPassword()));
        config.setDatabase(env.getRedisDb());
        return config;
    }

    private static JedisClientConfiguration.JedisClientConfigurationBuilder applyProperties(JedisClientConfiguration.JedisClientConfigurationBuilder builder) {
        Duration timeout = Duration.ofSeconds(4);
        builder.readTimeout(timeout).connectTimeout(timeout);
        return builder;
    }

    private static JedisPoolConfig jedisPoolConfig() {
        JedisPoolConfig config = new JedisPoolConfig();
        config.setMaxTotal(1);
        config.setMaxIdle(1);
        config.setMinIdle(1);
        config.setMaxWait(Duration.ofSeconds(5));
        return config;
    }
}
