package com.digiwin.dap.middleware.service.impl;

import com.digiwin.dap.middleware.domain.DapEnv;
import com.digiwin.dap.middleware.entity.BaseEntity;
import com.digiwin.dap.middleware.exception.BusinessException;
import com.digiwin.dap.middleware.repository.BaseEntityRepository;
import com.digiwin.dap.middleware.service.EntityManagerService;
import com.digiwin.dap.middleware.util.ObjectCopyUtil;
import com.digiwin.dap.middleware.util.TokenUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import javax.persistence.Table;
import java.lang.reflect.ParameterizedType;
import java.util.List;
import java.util.Optional;

/**
 * @author chenxsa
 */
@Service
public abstract class BaseEntityManagerService<T extends BaseEntity> implements EntityManagerService<T> {

    @Autowired
    protected DapEnv dapEnv;

    private String tableName;

    /**
     * 获取映射的表
     *
     * @return 表名
     */
    protected String getMappingTableName() {
        if (StringUtils.hasLength(tableName)) {
            return tableName;
        }
        Class<T> clazz = (Class<T>) ((ParameterizedType) getClass().getGenericSuperclass()).getActualTypeArguments()[0];
        Table table = clazz.getAnnotation(Table.class);
        if (table == null || !StringUtils.hasLength(table.name())) {
            tableName = clazz.getSimpleName();
        } else {
            if (table.name().startsWith("[")) {
                tableName = table.name().replace("[", "`");
                tableName = tableName.replace("]", "`");
            } else {
                tableName = table.name();
            }
        }
        return tableName;
    }

    protected abstract JpaRepository getRepository();

    @Override
    public void validate(T entity) {
    }

    @Override
    public long create(T entity) {
        TokenUtils.setCreateFields(entity);
        this.getRepository().save(entity);
        this.getRepository().flush();
        return entity.getSid();
    }

    @Override
    public void deleteById(long sid) {
        this.getRepository().deleteById(sid);
    }

    @Override
    public void update(T entity) {
        this.update(entity, false);
    }

    @Override
    public void update(T entity, boolean force) {
        if (entity.getSid() == 0) {
            throw new BusinessException("数据sid为0，无法进行更新，确认是否为保存");
        }
        TokenUtils.setModifyFields(entity);
        Optional<T> result = this.getRepository().findById(entity.getSid());
        if (result.isPresent()) {
            T dbData = result.get();
            ObjectCopyUtil.mergeObject(entity, dbData);
            this.getRepository().save(dbData);
            this.getRepository().flush();
        } else {
            throw new BusinessException("数据已经被删除.sid:" + entity.getSid());
        }
    }

    @Override
    public T findBySid(long sid) {
        Optional<T> entity = this.getRepository().findById(sid);
        return entity.orElse(null);
    }

    @Override
    public boolean exists(long sid) {
        return this.getRepository().existsById(sid);
    }

    @Override
    public void disable(long sid) {
        T result = this.findBySid(sid);
        if (result != null) {
            result.setDisabled(true);
            this.update(result);
        } else {
            throw new BusinessException("数据已经被删除.sid:" + sid);
        }
    }

    @Override
    public void enable(long sid) {
        T result = this.findBySid(sid);
        if (result != null) {
            result.setDisabled(false);
            this.update(result);
        } else {
            throw new BusinessException("数据已经被删除.sid:" + sid);
        }
    }

    @Override
    public void insertAll(Iterable<T> entities) {
        if (this.getRepository() instanceof BaseEntityRepository) {
            for (T entity : entities) {
                TokenUtils.setCreateFields(entity);
            }
            ((BaseEntityRepository) this.getRepository()).insert(entities);
        } else {
            this.saveAll(entities);
        }
    }

    @Override
    public void updateAll(Iterable<T> entities) {
        if (this.getRepository() instanceof BaseEntityRepository) {
            for (T entity : entities) {
                TokenUtils.setModifyFields(entity);
            }
            ((BaseEntityRepository) this.getRepository()).update(entities);
        } else {
            this.saveAll(entities);
        }
    }

    @Override
    public void saveAll(Iterable<T> entities) {
        for (T entity : entities) {
            if (entity.getSid() == 0) {
                TokenUtils.setCreateFields(entity);
            } else {
                TokenUtils.setModifyFields(entity);
            }
        }
        this.getRepository().saveAll(entities);
    }

    @Override
    public List<T> findAll() {
        return (List<T>) this.getRepository().findAll();
    }
}
