package com.digiwin.dap.middleware.converter;

import com.digiwin.dap.middleware.domain.SafeStorageValue;
import com.digiwin.dap.middleware.util.SecureUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.persistence.Converter;
import java.util.function.Function;

/**
 * 全局默认加解密转换器，针对所有{@link javax.persistence.Entity}PO中使用{@link SafeStorageValue}声明的字段，建议声明泛型，比如手机号就是整型
 * 这里类签名{@link SafeStorageValue}的入参无法声明泛型
 * TIPS: 在使用属性copy的地方需要手动设置
 * <p>
 * ==============
 * <p>
 * example-entity:<p>
 * <pre>
 *     &#064;Column(name = "name", columnDefinition = "VARCHAR(50) NOT NULL COMMENT '名称'")
 *     private SafeStorageValue<Date> name;
 *     </pre>
 * example-execute:
 * <pre>
 *         ConfigurableApplicationContext run = SpringApplication.run(GmcApplication.class, args);
 *         CategoryRepository bean = run.getBean(CategoryRepository.class);
 *         Category category = new Category();
 *         category.setCode("code");
 *         category.setName(new SafeStorageValue<>(new Date(), new Function<Date, String>() {
 *             &#064;Override
 *             public String apply(Date date) {
 *                 return date.toString();
 *             }
 *         }));
 *         Category save = bean.save(category);
 *         Date value = save.getName().toValue(new Function<String,Date>() {
 *             &#064;Override
 *             public Date apply(String o) {
 *                 //format
 *                 return new Date();
 *             }
 *         });
 * </pre>
 *
 * @author blockWilling
 * @date 2024/4/11 9:24
 * @mail kangjin@digiwin.com
 */
@Converter(autoApply = true)
public class AutoEnAndDecryptConverter extends BaseEnAndDecryptConverter<SafeStorageValue> {
    private static final Logger logger = LoggerFactory.getLogger(AutoEnAndDecryptConverter.class);

    @Override
    @Nullable
    public String convertToDatabaseColumn(SafeStorageValue attribute) {
        if (attribute == null) {
            return null;
        }
        Object value = attribute.get();
        if (value == null) {
            return null;
        }
        String toEncrypt = null;
        if (value instanceof String) {
            toEncrypt = (String) value;
        } else {
            Function<Object, String> toStringFunc = attribute.getToStringFunc();
            if (toStringFunc == null) {
                toEncrypt = value.toString();
            } else {
                toEncrypt = toStringFunc.apply(value);
            }
        }
        try {
            return SecureUtils.aesEncrypt(toEncrypt);
        } catch (Exception e) {
            logger.error("【SecureUtils.aesEncrypt】{}", e.getMessage());
            return toEncrypt;
        }
    }

    @Override
    @Nonnull
    public SafeStorageValue convertToEntityAttribute(String dbData) {
        if (dbData == null) {
            return new SafeStorageValue(null);
        }
        String aesDecrypt = null;
        try {
            aesDecrypt = SecureUtils.aesDecrypt(dbData);
        } catch (Exception e) {
            logger.error("【SecureUtils.aesDecrypt】{}", e.getMessage());
            return new SafeStorageValue(dbData);
        }
        //无法声明泛型，因为不知道具体的字段类型
        return new SafeStorageValue(aesDecrypt);
    }
}
