package com.digiwin.dap.middleware.service.impl;

import com.digiwin.dap.middleware.entity.AssociationEntity;
import com.digiwin.dap.middleware.entity.RelationAssociationEntity;
import com.digiwin.dap.middleware.exception.BusinessException;
import com.digiwin.dap.middleware.service.AssociationEntityService;
import com.digiwin.dap.middleware.service.CascadeDeleteEntityService;
import com.digiwin.dap.middleware.service.RelationAssociationEntityService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import java.util.Collections;
import java.util.List;
import java.util.Set;

/**
 * 级联删除服务
 *
 * @author fobgochod
 * @date 2021/1/12
 */
@Service
public class CascadeDeleteEntityServiceImpl implements CascadeDeleteEntityService {

    @Autowired
    private AssociationEntityService associationEntityService;
    @Autowired
    private RelationAssociationEntityService relationAssociationEntityService;

    /**
     * 注入的是实体管理器,执行持久化操作
     */
    @PersistenceContext
    private EntityManager entityManager;

    /**
     * 逗号,拼接sid
     * Object object = sids.stream().map(String::valueOf).collect(Collectors.joining(","));
     *
     * @param sids
     * @return
     */
    private static String getSids(List sids) {
        StringBuilder stringBuffer = new StringBuilder();
        for (int i = 0; i < sids.size(); i++) {
            if (i != sids.size() - 1) {
                stringBuffer.append(sids.get(i)).append(",");
            } else {
                stringBuffer.append(sids.get(i));
            }
        }
        return stringBuffer.toString();
    }

    /**
     * 根据表名和SID 删除与所有关联中的关联数据
     *
     * @param name 表名
     * @param sids sid
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void delete(String name, List<Long> sids) {
        //获取关联当前表的所有其他关联表
        Set<AssociationEntity> entities = associationEntityService.getAssociationEntities(name);
        for (AssociationEntity entity : entities) {
            //根据当前表的sid来获取依赖当前表的关联表中的所有主键
            List<Long> childSids = this.getSidByTableNameAndField(entity.getTableName(), entity.getFieldName(), sids);
            if (!childSids.isEmpty()) {
                if (entity.isCascadeDelete()) {
                    //如果是级联删除，继续删除
                    this.delete(entity.getTableName(), childSids);
                    this.deleteByTableNameAndField(entity.getTableName(), entity.getFieldName(), sids);
                } else {
                    //如果不能级联删除，则需要抛出异常
                    throw new BusinessException(String.format("不能删除表%s,因为表%s引用了主键：%s", name, entity.getTableName(), sids));
                }
            }
        }
    }

    /**
     * 根據表名和masteSid, slaveSid 刪除必須有此關連才能有其他關連的數據
     *
     * @param tableName 表名
     * @param masterSid
     * @param slaveSid
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void deleteRelation(String tableName, Long masterSid, Long slaveSid) {
        List<RelationAssociationEntity> entities = relationAssociationEntityService.getAssociationEntities(tableName);
        entities.forEach((entity) -> {
            if (entity.isCheckSelf()) {
                deleteRelationByEntityAndCheckSelf(entity, masterSid, slaveSid);
            } else {
                deleteRelationByEntity(entity, masterSid, slaveSid);
            }
        });
    }

    /**
     * select sid from tableName where fieldName in (sids)
     *
     * @param tableName 表名
     * @param fieldName 字段名
     * @param sids      字段值列表
     * @return Sid 列表
     */
    private List getSidByTableNameAndField(String tableName, String fieldName, List<Long> sids) {
        if (!StringUtils.hasLength(fieldName)) {
            return Collections.emptyList();
        }
        String sql = String.format("SELECT SID FROM `%s` WHERE %s IN (%s)", tableName, fieldName, getSids(sids));
        return entityManager.createNativeQuery(sql).getResultList();
    }

    /**
     * delete from tableName where fieldName in (sids)
     *
     * @param tableName 表名
     * @param fieldName 字段名
     * @param sids      字段值列表
     */
    private void deleteByTableNameAndField(String tableName, String fieldName, List<Long> sids) {
        if (!StringUtils.hasLength(fieldName)) {
            return;
        }
        String sql = String.format("DELETE FROM `%s` WHERE %s IN (%s)", tableName, fieldName, getSids(sids));
        entityManager.createNativeQuery(sql).executeUpdate();
    }

    /**
     * delete from tableName where masterField = masterSid and slaveField in (select sid from slaveName where checkFiled = slaveSid)
     *
     * @param entity
     * @param masterSid
     * @param slaveSid
     */
    private void deleteRelationByEntity(RelationAssociationEntity entity, Long masterSid, Long slaveSid) {
        String sql = String.format("DELETE FROM `%s` WHERE `%s` = %s AND `%s` IN (SELECT sid FROM `%s` WHERE `%s` = %s)",
                entity.getTableName(),
                entity.getMasterField(), masterSid,
                entity.getSlaveField(),
                entity.getSlaveTable(), entity.getCheckField(), slaveSid);
        entityManager.createNativeQuery(sql).executeUpdate();
    }

    /**
     * delete from tableName where masterField = masterSid and checkField = slaveSid
     *
     * @param entity
     * @param masterSid
     * @param slaveSid
     */
    private void deleteRelationByEntityAndCheckSelf(RelationAssociationEntity entity, Long masterSid, Long slaveSid) {
        String sql = String.format("DELETE FROM `%s` WHERE `%s` = %s AND `%s` = %s",
                entity.getTableName(),
                entity.getMasterField(), masterSid,
                entity.getCheckField(), slaveSid);
        entityManager.createNativeQuery(sql).executeUpdate();
    }
}
