package com.digiwin.dap.middleware.support.upgrade;

import com.digiwin.dap.middleware.auth.AuthoredUser;
import com.digiwin.dap.middleware.exception.BusinessException;
import com.digiwin.dap.middleware.service.UpdateDatabaseService;
import com.digiwin.dap.middleware.util.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.core.JdbcTemplate;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

/**
 * 版本升级服务
 *
 * @author fobgochod
 * @date 2019/10/18
 */
public class DefaultUpgradeHandler implements UpgradeHandler {

    private static final Logger logger = LoggerFactory.getLogger(DefaultUpgradeHandler.class);
    private final JdbcTemplate jdbcTemplate;
    private final List<UpdateDatabaseService> updateDatabaseServices;

    public DefaultUpgradeHandler(JdbcTemplate jdbcTemplate, List<UpdateDatabaseService> updateDatabaseServices) {
        this.jdbcTemplate = jdbcTemplate;
        this.updateDatabaseServices = updateDatabaseServices;
    }

    private void createTableVersion() {
        try {
            boolean exists = jdbcTemplate.queryForList("show tables like 'version'").stream().findAny().isPresent();
            if (!exists) {
                String sql = FileUtil.getFile("/support/version.sql");
                jdbcTemplate.execute(sql);
            }
        } catch (Exception e) {
            logger.error("create table version error", e);
        }
    }

    @Override
    public List<Object> upgrade(String startVersion, String endVersion) {
        List<Object> upgrades = new ArrayList<>();
        this.createTableVersion();

        int oldVersion = jdbcTemplate.queryForList("select version from version order by sid desc limit 1", Integer.class)
                .stream()
                .findFirst()
                .orElse(0) + 1;
        for (UpdateDatabaseService currentService : updateDatabaseServices) {
            if (currentService.support(startVersion, endVersion)) {
                long startTime = System.currentTimeMillis();
                currentService.before();
                currentService.update();
                currentService.after();
                long upgradeTime = System.currentTimeMillis() - startTime;
                save(oldVersion, currentService.version(), upgradeTime);
                upgrades.add(new UpgradeVO(upgradeTime, currentService.version(), currentService.getClass().getName()));
            }
        }
        return upgrades;
    }

    @Override
    public Version findLatestVersion() {
        List<Version> lists = jdbcTemplate.query("select sid,success,version,middleware_version,update_by,update_by_id,update_date,execution_time from version where success = 1 order by sid desc limit 1", new VersionMapper());
        return lists.stream().findFirst().orElse(null);
    }

    private void save(int oldVersion, String middlewareVersion, long executionTime) {
        AuthoredUser authoredUser = UserUtils.getAuthoredUser();
        String insert;
        if (authoredUser == null) {
            insert = String.format("INSERT INTO version(sid,success,version,middleware_version,update_date,execution_time) " + "VALUES (%s, true, %s, '%s', '%s', %s);", SnowFlake.getInstance()
                    .newId(), oldVersion, middlewareVersion, DateUtils.formatDateTime(LocalDateTime.now()), executionTime);
        } else {
            insert = String.format("INSERT INTO version(sid,success,version,middleware_version,update_by,update_by_id,update_date,execution_time) " + "VALUES (%s, true, %s, '%s', %s, '%s', '%s', %s);", SnowFlake.getInstance()
                    .newId(), oldVersion, middlewareVersion, authoredUser.getSid(), authoredUser.getUserName(), DateUtils.formatDateTime(LocalDateTime.now()), executionTime);

        }
        jdbcTemplate.execute(insert);
    }
}
