package com.digiwin.dap.middleware.service.impl;

import com.digiwin.dap.middleware.entity.BaseEntityWithIdAndTenant;
import com.digiwin.dap.middleware.exception.BusinessException;
import com.digiwin.dap.middleware.repository.BaseEntityWithTenantRepository;
import com.digiwin.dap.middleware.service.EntityWithTenantManagerService;

import jakarta.persistence.EntityManager;
import jakarta.persistence.PersistenceContext;
import jakarta.persistence.Query;
import java.util.List;

/**
 * 表实体服务基类，提供基于id和租户sid为不可重复索引的表基本操作
 */
public abstract class BaseEntityWithTenantManagerService<T extends BaseEntityWithIdAndTenant> extends BaseEntityManagerService<T> implements EntityWithTenantManagerService<T> {


    @PersistenceContext
    EntityManager entityManager;

    /**
     * 获取jpa的rep
     *
     * @return
     */
    @Override
    protected abstract BaseEntityWithTenantRepository getRepository();

    /**
     * 根据tenantSid,id获取sid
     *
     * @param id
     * @return sid
     */
    @Override
    public long getSidByTenantAndId(long tenantSid, String id) {
        String sql = String.format("SELECT sid FROM %s WHERE tenant_sid = :tenantSid AND id = :id LIMIT 1", this.getMappingTableName());
        Query query = entityManager.createNativeQuery(sql);
        query.setParameter("tenantSid", tenantSid);
        query.setParameter("id", id);

        List resultList = query.getResultList();
        if (resultList.size() == 0) {
            return 0;
        }
        return Long.parseLong(resultList.get(0).toString());
    }

    /**
     * 根据实体Id 和租户Id 查询实体
     *
     * @param tenantSid 租户Sid
     * @param id        实体Id
     * @return 实体对象
     */
    @Override
    public T findByTenantSidAndId(long tenantSid, String id) {
        return (T) getRepository().findByTenantSidAndId(tenantSid, id);
    }

    /**
     * 根据实体Id 和租户Id 删除实体
     *
     * @param id        实体Id
     * @param tenantSid 租户Sid
     */
    @Override
    public void deleteByTenantSidAndId(long tenantSid, String id) {
        getRepository().deleteByTenantSidAndId(tenantSid, id);
    }

    /**
     * 根据租户sid和id判断是否存在
     *
     * @param tenantSid
     * @param id
     * @return
     */
    @Override
    public boolean existsByTenantSidAndId(long tenantSid, String id) {
        return this.getRepository().existsByTenantSidAndId(tenantSid, id);
    }


    /**
     * 启用
     *
     * @param tenantSid 租户sid
     * @param id        实体id
     * @return
     */
    @Override
    public void enable(long tenantSid, String id) {
        T result = (T) this.getRepository().findByTenantSidAndId(tenantSid, id);
        if (result != null) {
            result.setDisabled(false);
            this.getRepository().save(result);
        } else {
            throw new BusinessException("数据已经被删除.tenantSid:" + tenantSid + ",id:" + id);
        }
    }

    /**
     * 禁用
     *
     * @param tenantSid 租户sid
     * @param id        实体id
     * @return
     */
    @Override
    public void disable(long tenantSid, String id) {
        T result = (T) this.getRepository().findByTenantSidAndId(tenantSid, id);
        if (result != null) {
            result.setDisabled(true);
            this.getRepository().save(result);
        } else {
            throw new BusinessException("数据已经被删除.tenantSid:" + tenantSid + ",id:" + id);
        }
    }
}
