package com.digiwin.dap.middleware.entity;

import com.digiwin.dap.middleware.commons.util.BeanUtils;
import com.digiwin.dap.middleware.commons.util.StrUtils;
import com.digiwin.dap.middleware.exception.BusinessException;
import jakarta.persistence.Column;
import jakarta.persistence.criteria.*;
import jakarta.validation.constraints.NotNull;
import org.springframework.util.Assert;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 描述业务主键
 *
 * @author fobgochod
 * @date 2020/4/27
 */
public class UnionKey {

    private static final Pattern regex = Pattern.compile("[A-Z]+");
    private final List<String> fields = new ArrayList<>();
    private final List<String> tableFields = new ArrayList<>();
    private Class<?> entityClass;

    public static UnionKey create() {
        return new UnionKey();
    }

    public List<String> getFields() {
        return fields;
    }

    public UnionKey apply(@NotNull Class<?> entityClass) {
        Assert.notNull(entityClass, "entityClass 不能为null");
        this.entityClass = entityClass;
        return this;
    }

    /**
     * 添加业务主键的字段。注意顺序
     *
     * @param name
     * @return
     */
    public UnionKey add(@NotNull String name) {
        if (StrUtils.isEmpty(name)) {
            throw new IllegalArgumentException("name 不能为空");
        }
        if (fields.contains(name)) {
            throw new IllegalArgumentException("存在重复的值：" + name);
        }
        Assert.notNull(entityClass, "entityClass 不能为null,请先调用apply方法");

        Field field = BeanUtils.getDeclaredField(this.entityClass, name);
        if (field == null) {
            throw new BusinessException(String.format("%s 上不存在属性%s", entityClass, name));
        }

        this.fields.add(name);
        String columnName = name;
        Column column = field.getAnnotation(Column.class);
        if (column != null && StrUtils.isNotEmpty(column.name())) {
            columnName = column.name().toUpperCase();
            columnName = columnName.replace("[", "`");
            columnName = columnName.replace("]", "`");
            this.tableFields.add(columnName);
        } else {

            Matcher matcher = UnionKey.regex.matcher(columnName);
            if (matcher.find()) {
                columnName = matcher.replaceAll("_$0");
            }
            columnName = columnName.toUpperCase();
            this.tableFields.add(columnName);
        }
        return this;
    }

    /**
     * 清除
     */
    public void clear() {
        this.fields.clear();
        this.tableFields.clear();
    }

    /**
     * 获取查询的sql
     */
    public String getSelectSql() {
        StringBuilder sql = new StringBuilder();
        for (int i = 0; i < tableFields.size(); i++) {
            sql.append(tableFields.get(i));
            if (i < tableFields.size() - 1) {
                sql.append(",");
            }
        }
        return sql.toString();
    }

    /**
     * 拼接动态sql
     *
     * @param values
     * @return TENANT_SID = :tenantSid AND ID = :id
     */
    public String getWhere(Object... values) {
        if (values == null || values.length != fields.size()) {
            throw new IllegalArgumentException("值不能为空或者可能小于或者大于唯一键的数量");
        }

        StringBuilder sql = new StringBuilder();
        for (int i = 0; i < tableFields.size(); i++) {
            sql.append(String.format("%s = :%s", tableFields.get(i), fields.get(i)));
            if (i < fields.size() - 1) {
                sql.append(" AND ");
            }
        }
        return sql.toString();
    }

    /**
     * 拼凑where条件
     *
     * @param values 如果值为null，也必须传入null
     */
    public String getWhereSql(Object... values) {
        if (values == null || values.length != fields.size()) {
            throw new IllegalArgumentException("值不能为空或者可能小于或者大于主键的数量");
        }

        StringBuilder sql = new StringBuilder();
        for (int i = 0; i < tableFields.size(); i++) {
            Object paraValue = values[i];
            String paraName = tableFields.get(i);

            if (paraValue == null) {
                sql.append(String.format("ISNULL(%s)", paraName));
            } else {

                if (paraValue instanceof Integer || paraValue instanceof Long || paraValue instanceof Double || paraValue instanceof Boolean) {
                    sql.append(String.format("%s = %s", paraName, paraValue));
                } else {
                    sql.append(String.format("%s = '%s'", paraName, paraValue));
                }
            }
            if (i < fields.size() - 1) {
                sql.append(" AND ");
            }
        }
        return sql.toString();
    }

    /**
     * 生成支持jpa自定义查询的查询类
     */
    public <E extends BaseEntity> Predicate getSpecification(Root<E> root,
                                                             CriteriaQuery<?> cq,
                                                             CriteriaBuilder cb,
                                                             Object... values) {
        if (values == null || values.length != fields.size()) {
            throw new IllegalArgumentException("值不能为空或者可能小于主键的数量");
        }

        List<Predicate> predicates = new ArrayList<>(fields.size());
        for (int i = 0; i < fields.size(); i++) {
            Object paraValue = values[i];
            String paraName = fields.get(i);
            Path<?> path = root.get(paraName);
            Class<?> attrType = path.getJavaType();

            if (paraValue == null) {
                predicates.add(cb.isNull(path));
                continue;
            }

            // 如果传入的是 String 且目标是 Enum，转换成 Enum
            if (attrType.isEnum()) {
                @SuppressWarnings({"unchecked", "rawtypes"})
                Enum enumVal = Enum.valueOf((Class<Enum>) attrType, paraValue.toString());
                predicates.add(cb.equal(path.as(attrType), enumVal));
                continue;
            }

            // 若目标是 Number 类型，尝试转换（支持 Integer, Long, Double, BigDecimal 等）
            if (Number.class.isAssignableFrom(attrType) || attrType.isPrimitive() && (
                    attrType == int.class || attrType == long.class || attrType == double.class || attrType == float.class)) {
                Number number = toNumber(paraValue, (Class<? extends Number>) attrType);
                predicates.add(cb.equal(path.as(number.getClass()), number));
                continue;
            }

            // 布尔类型
            if (attrType == Boolean.class || attrType == boolean.class) {
                Boolean b = Boolean.valueOf(paraValue.toString());
                predicates.add(cb.equal(path.as(Boolean.class), b));
                continue;
            }

            // 日期/时间类型（示例：LocalDate/LocalDateTime/Instant/Date），按需要扩展
            if (java.util.Date.class.isAssignableFrom(attrType)) {
                // 如果传入是 Number(long epoch) 或 String，做相应转换
                java.util.Date d = toDate(paraValue);
                predicates.add(cb.equal(path.as(java.util.Date.class), d));
                continue;
            }

            // 默认：按 String 比较（注意：如果 DB 列是 varchar，但实际对象类型不是 String，path.as会转换）
            predicates.add(cb.equal(path.as(String.class), paraValue.toString()));
        }

        return predicates.isEmpty() ? cb.conjunction() : cb.and(predicates.toArray(new Predicate[0]));
    }

    // 辅助：将 paraValue 转为目标 Number 类型
    private Number toNumber(Object value, Class<? extends Number> target) {
        if (value instanceof Number) {
            Number n = (Number) value;
            if (target == Integer.class || target == int.class) return n.intValue();
            if (target == Long.class || target == long.class) return n.longValue();
            if (target == Double.class || target == double.class) return n.doubleValue();
            if (target == Float.class || target == float.class) return n.floatValue();
            if (target == Short.class || target == short.class) return n.shortValue();
            if (target == Byte.class || target == byte.class) return n.byteValue();
            // fallback
            return n;
        }
        String s = value.toString();
        if (target == Integer.class || target == int.class) return Integer.valueOf(s);
        if (target == Long.class || target == long.class) return Long.valueOf(s);
        if (target == Double.class || target == double.class) return Double.valueOf(s);
        if (target == Float.class || target == float.class) return Float.valueOf(s);
        if (target == Short.class || target == short.class) return Short.valueOf(s);
        if (target == Byte.class || target == byte.class) return Byte.valueOf(s);
        // fallback BigDecimal
        return new java.math.BigDecimal(s);
    }

    // 简单日期转换示例（按你项目需求扩展）
    private java.util.Date toDate(Object value) {
        if (value instanceof java.util.Date) return (java.util.Date) value;
        if (value instanceof Number) return new java.util.Date(((Number) value).longValue());
        // try parse ISO string
        try {
            java.time.Instant inst = java.time.Instant.parse(value.toString());
            return java.util.Date.from(inst);
        } catch (Exception e) {
            // parse failed，抛异常或返回 null 视需
            throw new IllegalArgumentException("无法将值转换为 Date: " + value, e);
        }
    }
}
