package com.digiwin.dap.middle.cache.limiter;

import com.digiwin.dap.middle.cache.limiter.callback.AbstractRateLimiterCallback;
import com.digiwin.dap.middle.cache.limiter.enums.RateLimitingDimensionEnum;
import com.digiwin.dap.middle.cache.limiter.enums.RateLimitingDimensionOperatorEnum;

import java.lang.annotation.*;

/**
 * <p>令牌桶算法</p>
 * <p>令牌桶算法：定期向桶中加入令牌，每次请求消耗一个令牌。当桶中没有令牌时，新的请求会被拒绝或等待</p>
 * <p>特点：允许短时间突发流量、高弹性的限流、带宽限制</p>
 * <p>优点：</p><p>1、可以支持一定程度的流量突发，避免在突发流量情况下直接拒绝请求</p>
 * <p>缺点：</p><p>1、当突发流量较大且持续时间较长时，可能会出现流量超标的情况；2、对于严格限制每秒请求次数的场景，可能不如滑动窗口精确</p>
 * <p>假设每个请求都需要一个令牌，接口处理能力=令牌生成速率</p>
 *
 * @author michael
 * @since 2.7.19.16
 */
@Documented
@Inherited
@Target(ElementType.METHOD)
@Retention(RetentionPolicy.RUNTIME)
public @interface RateLimiter {

    /**
     * token总数
     */
    int burstCapacity() default 1;

    /**
     * 每个请求消耗几个token
     */
    int requestedTokens() default 1;

    /**
     * token生成速度,单位:个/秒
     */
    int replenishRate() default 1;

    /**
     * 限流自定义参数,支持SpEL,当限流维度设置CUSTOM_PARAM时使用
     */
    String param() default "";

    /**
     * 限流回调
     */
    Class<? extends AbstractRateLimiterCallback> callback() default AbstractRateLimiterCallback.class;

    /**
     * 限流维度
     */
    RateLimitingDimensionEnum[] dimensions() default {RateLimitingDimensionEnum.TENANT};

    /**
     * 多个限流维度运算符
     */
    RateLimitingDimensionOperatorEnum operator() default RateLimitingDimensionOperatorEnum.AND;
}
