package com.digiwin.dap.middleware.cache;

import com.digiwin.dap.middleware.util.JsonUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.redis.connection.RedisStandaloneConfiguration;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.data.redis.core.*;

import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * RedisCache.java
 *
 * @author fobgochod
 * @date 2021/8/19 16:15
 */
public class RedisUtils {

    private static final Logger logger = LoggerFactory.getLogger(RedisUtils.class);
    private static final ObjectMapper objectMapper = JsonUtils.createObjectMapper();
    private static final long DEFAULT_LOCK_TIME = 120;
    private static final long REDIS_SCAN_COUNT = 50000;
    private static final String REDIS_ERROR = "【Redis异常】";

    private static RedisTemplate<String, Object> redisTemplate;

    public static void initTemplate(RedisTemplate<String, Object> redisTemplate) {
        RedisUtils.redisTemplate = redisTemplate;
    }

    public static RedisTemplate<String, Object> getRedisTemplate() {
        return redisTemplate;
    }

    public static ValueOperations<String, Object> opsForValue() {
        return redisTemplate.opsForValue();
    }

    public static HashOperations<String, Object, Object> opsForHash() {
        return redisTemplate.opsForHash();
    }

    public static ListOperations<String, Object> opsForList() {
        return redisTemplate.opsForList();
    }

    public static SetOperations<String, Object> opsForSet() {
        return redisTemplate.opsForSet();
    }

    public static ZSetOperations<String, Object> opsForZSet() {
        return redisTemplate.opsForZSet();
    }

    public static void set(String key, Object value) {
        try {
            String json = objectMapper.writeValueAsString(value);
            redisTemplate.opsForValue().set(key, json);
        } catch (Exception e) {
            logger.error("{}Key值为:{}，错误信息：{}", REDIS_ERROR, key, e.getMessage());
        }
    }

    public static void set(String key, Object value, Duration duration) {
        try {
            String json = objectMapper.writeValueAsString(value);
            redisTemplate.opsForValue().set(key, json, duration);
        } catch (Exception e) {
            logger.error("{}Key值为:{}，错误信息：{}", REDIS_ERROR, key, e.getMessage());
        }
    }

    public static void delete(String key) {
        try {
            redisTemplate.delete(key);
        } catch (Exception e) {
            logger.error("{}Key值为:{}，错误信息：{}", REDIS_ERROR, key, e.getMessage());
        }
    }

    public static Long delete(Collection<String> keys) {
        try {
            return redisTemplate.delete(keys);
        } catch (Exception e) {
            logger.error("【Redis异常】Key值为:{}，错误信息：{}", keys, e.getMessage());
        }
        return 0L;
    }

    public static <T> T get(Object key, Class<T> type) {
        try {
            Object obj = redisTemplate.opsForValue().get(key);
            if (obj != null) {
                return objectMapper.readValue(obj.toString(), type);
            }
        } catch (Exception e) {
            logger.error("{}Key值为:{}，错误信息：{}", REDIS_ERROR, key, e.getMessage());
        }
        return null;
    }

    public static <T> T get(Object key, TypeReference<T> valueTypeRef) {
        try {
            Object obj = redisTemplate.opsForValue().get(key);
            if (obj != null) {
                return objectMapper.readValue(obj.toString(), valueTypeRef);
            }
        } catch (Exception e) {
            logger.error("{}Key值为:{}，错误信息：{}", REDIS_ERROR, key, e.getMessage());
        }
        return null;
    }

    /**
     * keys 模糊查询，已替换为 scan
     * <p>
     * 推荐使用 scan
     *
     * @param pattern 示例 iam:token:*
     * @return the keys Set
     * @see #scan(String)
     */
    public static Set<String> keys(String pattern) {
        try {
            return scan(pattern);
        } catch (Exception e) {
            logger.error("【Redis异常】Key值为:{}，错误信息：{}", pattern, e.getMessage());
        }
        return Collections.emptySet();
    }

    /**
     * scan 模糊查询
     *
     * @param pattern 示例 iam:token:*
     * @return the keys Set
     */
    public static Set<String> scan(String pattern) {
        return redisTemplate.execute((RedisCallback<Set<String>>) connection -> {
            Set<String> result = new HashSet<>();
            try (Cursor<byte[]> cursor = connection.scan(ScanOptions.scanOptions().match(pattern).count(REDIS_SCAN_COUNT).build())) {
                while (cursor.hasNext()) {
                    result.add(new String(cursor.next(), StandardCharsets.UTF_8));
                }
            } catch (Exception e) {
                logger.error("【Redis异常】Key值为:{}，错误信息：{}", pattern, e.getMessage());
            }
            return result;
        });
    }

    public static boolean hasKey(String key) {
        try {
            Boolean result = redisTemplate.hasKey(key);
            return Objects.equals(result, Boolean.TRUE);
        } catch (Exception e) {
            logger.error("{}Key值为:{}，错误信息：{}", REDIS_ERROR, key, e.getMessage());
        }
        return false;
    }

    public static void expire(String key, long timeout, TimeUnit unit) {
        try {
            redisTemplate.expire(key, timeout, unit);
        } catch (Exception e) {
            logger.error("{}Key值为:{}，错误信息：{}", REDIS_ERROR, key, e.getMessage());
        }
    }

    public static Long getExpire(String key) {
        try {
            return redisTemplate.getExpire(key);
        } catch (Exception e) {
            logger.error("{}Key值为:{}，错误信息：{}", REDIS_ERROR, key, e.getMessage());
        }
        return null;
    }

    public static Long getExpire(String key, TimeUnit timeUnit) {
        try {
            return redisTemplate.getExpire(key, timeUnit);
        } catch (Exception e) {
            logger.error("{}Key值为:{}，错误信息：{}", REDIS_ERROR, key, e.getMessage());
        }
        return null;
    }

    public static boolean lock(String key, String value) {
        try {
            Boolean result = redisTemplate.opsForValue().setIfAbsent(key, value, DEFAULT_LOCK_TIME, TimeUnit.SECONDS);
            return Objects.equals(result, Boolean.TRUE);
        } catch (Exception e) {
            logger.error("{}Key值为:{}，错误信息：{}", REDIS_ERROR, key, e.getMessage());
            return false;
        }
    }

    public static void unlock(String key) {
        try {
            redisTemplate.opsForValue().getOperations().delete(key);
        } catch (Exception e) {
            logger.error("{}Key值为:{}，错误信息：{}", REDIS_ERROR, key, e.getMessage());
        }
    }

    public static boolean setIfAbsent(String key, Object value, Duration timeout) {
        try {
            Boolean absent = redisTemplate.opsForValue().setIfAbsent(key, value, timeout);
            return Objects.equals(absent, Boolean.TRUE);
        } catch (Exception e) {
            logger.error("{}Key值为:{}，错误信息：{}", REDIS_ERROR, key, e.getMessage());
        }
        return false;
    }

    public static boolean setIfAbsent(String key, Object value, long timeout, TimeUnit unit) {
        try {
            Boolean absent = redisTemplate.opsForValue().setIfAbsent(key, value, timeout, unit);
            return Objects.equals(absent, Boolean.TRUE);
        } catch (Exception e) {
            logger.error("{}Key值为:{}，错误信息：{}", REDIS_ERROR, key, e.getMessage());
        }
        return false;
    }

    public static void changeDb(int num) {
        JedisConnectionFactory jedisConnectionFactory = (JedisConnectionFactory) redisTemplate.getConnectionFactory();
        if (jedisConnectionFactory != null && jedisConnectionFactory.getDatabase() != num) {
            RedisStandaloneConfiguration standaloneConfig = jedisConnectionFactory.getStandaloneConfiguration();
            if (standaloneConfig != null) {
                standaloneConfig.setDatabase(num);
                redisTemplate.setConnectionFactory(jedisConnectionFactory);
            }
        }
    }
}
