package com.digiwin.dap.middle.cache.limiter.aspect;

import com.digiwin.dap.middle.cache.limiter.RateLimiter;
import com.digiwin.dap.middle.cache.limiter.RateLimiterKeyGenerators;
import com.digiwin.dap.middle.cache.limiter.callback.RateLimiterCallback;
import com.digiwin.dap.middle.cache.limiter.constants.RateLimiterConstant;
import com.digiwin.dap.middleware.domain.CommonErrorCode;
import com.digiwin.dap.middleware.exception.RequestNotPermittedException;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.script.RedisScript;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author michael
 * @since 2.7.19.16
 */
@Aspect
@Component
public class TokenBucketRateLimiterInterceptor {

    private static final Logger logger = LoggerFactory.getLogger(TokenBucketRateLimiterInterceptor.class);

    private final RedisTemplate<String, Object> redisTemplate;
    private final RedisScript<Boolean> script;

    @Value("${spring.application.name:dap}")
    private String appName;

    @Value("${dap.middleware.rate.limiter:true}")
    private boolean rateLimiter;

    public TokenBucketRateLimiterInterceptor(RedisTemplate<String, Object> redisTemplate, @Qualifier(RateLimiterConstant.TOKEN_BUCKET_RATE_LIMITER_SCRIPT_NAME) RedisScript<Boolean> script) {
        this.redisTemplate = redisTemplate;
        this.script = script;
    }

    @Before("(@annotation(com.digiwin.dap.middle.cache.limiter.RateLimiter))")
    public void execute(JoinPoint joinPoint) {
        if(!rateLimiter) {
            return;
        }
        Boolean allowed = true;
        Class<?> callbackClass = null;
        try {
            RateLimiter rateLimiter = ((MethodSignature) joinPoint.getSignature()).getMethod().getAnnotation(RateLimiter.class);
            int replenishRate = rateLimiter.replenishRate();
            int burstCapacity = rateLimiter.burstCapacity();
            int requestedTokens = rateLimiter.requestedTokens();
            callbackClass = rateLimiter.callback();

            List<String> keyList = RateLimiterKeyGenerators.generateRateLimiterKey(joinPoint.getArgs(), joinPoint.getTarget(),
                    joinPoint.getSignature(), rateLimiter.param(), rateLimiter.dimensions(),
                    rateLimiter.operator());
            if (CollectionUtils.isEmpty(keyList)) {
                throw new RequestNotPermittedException(CommonErrorCode.TOO_MANY_REQUESTS);
            }
            keyList = keyList.stream()
                    .map(element -> appName + ":" + RateLimiterConstant.TOKEN_BUCKET_PREFIX + element)
                    .collect(Collectors.toList());
            for (String key : keyList) {
                allowed = redisTemplate.execute(this.script, Collections.singletonList(key), burstCapacity, replenishRate, requestedTokens);
                if (allowed == null || !allowed) {
                    break;
                }
            }

        } catch (Exception e) {
            logger.error("Error determining if user allowed from redis", e);
        }
        if (allowed == null || !allowed) {
            RateLimiterCallback rateLimiterCallback = null;
            try {
                rateLimiterCallback = (RateLimiterCallback) callbackClass.newInstance();
            } catch (Exception e) {
                logger.error("rate limited callback error", e);
            }
            if (rateLimiterCallback != null) {
                rateLimiterCallback.onRateLimited();
            } else {
                throw new RequestNotPermittedException(CommonErrorCode.TOO_MANY_REQUESTS);
            }
        }
    }
}
