package com.digiwin.dap.middle.database.encrypt.sql.parser.impl;

import com.digiwin.dap.middle.database.encrypt.enums.DatabaseEncryptExceptionEnum;
import com.digiwin.dap.middle.database.encrypt.exception.DatabaseEncryptException;
import com.digiwin.dap.middle.database.encrypt.model.ObjectRelationalMapping;
import com.digiwin.dap.middle.database.encrypt.sql.parser.SqlParser;
import com.digiwin.dap.middleware.commons.util.StrUtils;
import net.sf.jsqlparser.JSQLParserException;
import net.sf.jsqlparser.expression.Expression;
import net.sf.jsqlparser.parser.CCJSqlParserUtil;
import net.sf.jsqlparser.schema.Column;
import net.sf.jsqlparser.schema.Table;
import net.sf.jsqlparser.statement.Statement;
import net.sf.jsqlparser.statement.select.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author michael
 */
public class CCJSqlParserImpl implements SqlParser {

    private final static Logger LOGGER = LoggerFactory.getLogger(CCJSqlParserImpl.class);
    private final static String MYSQL_DIALECT = "`";

    private final Map<String, List<ObjectRelationalMapping>> mappingMap = new ConcurrentHashMap<>();

    private static List<ObjectRelationalMapping> parseResultSetFromStatement(Select select, List<CCJSqlParserImpl.TableInfo> tableInfoList) {
        List<ObjectRelationalMapping> objectRelationalMappings = new ArrayList<>();
        select.accept(new SelectVisitorAdapter<Void>() {
            @Override
            public <S> Void visit(PlainSelect plainSelect, S context) {
                List<SelectItem<?>> selectItems = plainSelect.getSelectItems();
                for (SelectItem<?> selectItem : selectItems) {
                    Expression expression = selectItem.getExpression();
                    if (expression instanceof Column) {
                        String expressionString = expression.toString();
                        String aliasTableName = expressionString.contains(".") ? expressionString.substring(0, expressionString.indexOf(".")) : "";
                        String columnName = expressionString.contains(".") ? expressionString.substring(expressionString.indexOf(".") + 1) : expressionString;
                        String aliasColumnName = selectItem.getAlias() != null ? selectItem.getAlias().getName() : columnName;
                        String tableName = StrUtils.isNotEmpty(aliasTableName) ? tableInfoList.stream().filter(x -> Objects.equals(x.getAliasTableName(), aliasTableName)).findFirst().get().getTableName().toLowerCase() :
                                tableInfoList.get(0).getTableName().toLowerCase();
                        // 去掉列名上的``
                        columnName = columnName.replaceAll(MYSQL_DIALECT, "");
                        aliasColumnName = aliasColumnName.replaceAll(MYSQL_DIALECT, "");
                        objectRelationalMappings.add(new ObjectRelationalMapping(tableName, columnName, aliasColumnName));
                        LOGGER.debug("====>数据库【{}】表,字段【{}】,映射到对象中属性名为【{}】", tableName, columnName, aliasColumnName);
                    } else {
                        LOGGER.debug("====>列【{}】暂不支持解析", expression);
                    }
                }
                return null;
            }
        }, null);
        return objectRelationalMappings;
    }

    private static List<CCJSqlParserImpl.TableInfo> parseTableFromStatement(Statement statement) {
        List<CCJSqlParserImpl.TableInfo> tableInfoList = new ArrayList<>();
        if (statement instanceof Select) {
            Select select = (Select) statement;
            select.accept(new SelectVisitorAdapter<Void>() {
                @Override
                public <S> Void visit(PlainSelect plainSelect, S context) {
                    FromItem fromItem = plainSelect.getFromItem();
                    if (fromItem instanceof Table) {
                        Table table = (Table) fromItem;
                        String tableName = table.getName();

                        String[] parts = tableName.split("\\s+");
                        tableName = parts[0];
                        // 去掉表名上的``
                        tableName = tableName.replaceAll(MYSQL_DIALECT, "");
                        String aliasTableName = table.getAlias() != null ? table.getAlias().getName() : tableName;
                        aliasTableName = aliasTableName.replaceAll(MYSQL_DIALECT, "");
                        tableInfoList.add(new CCJSqlParserImpl.TableInfo(tableName, aliasTableName));
                    }

                    if (plainSelect.getJoins() != null) {
                        for (Join join : plainSelect.getJoins()) {
                            FromItem joinItem = join.getRightItem();
                            if (joinItem instanceof Table) {
                                Table table = (Table) joinItem;
                                String tableName = table.getName();
                                String[] parts = tableName.split("\\s+");
                                tableName = parts[0];
                                // 去掉表名上的``
                                tableName = tableName.replaceAll(MYSQL_DIALECT, "");
                                String aliasTableName = table.getAlias() != null ? table.getAlias().getName() : tableName;
                                aliasTableName = aliasTableName.replaceAll(MYSQL_DIALECT, "");
                                tableInfoList.add(new CCJSqlParserImpl.TableInfo(tableName, aliasTableName));
                            }
                        }
                    }

                    return null;
                }
            }, null);
        }
        return tableInfoList;
    }

    @Override
    public List<ObjectRelationalMapping> parseQuerySql(String id, String sql) {
        List<ObjectRelationalMapping> objectRelationalMappings = new ArrayList<>();
        if (mappingMap.containsKey(id)) {
            return mappingMap.get(id);
        }
        try {
            Statement statement = CCJSqlParserUtil.parse(sql);
            if (!(statement instanceof Select)) {
                LOGGER.debug("====>sql解析,暂不支持解析非select语句");
                return objectRelationalMappings;
            }
            // 1.获取sql中出现的表名以及其别名的映射关系
            List<CCJSqlParserImpl.TableInfo> tableInfoList = parseTableFromStatement(statement);
            // 2.select语句获取结果集表名、表别名、字段名、字段别名的映射关系注意
            objectRelationalMappings = parseResultSetFromStatement((Select) statement, tableInfoList);
        } catch (Exception e) {
            if (e instanceof JSQLParserException) {
                LOGGER.error("====>解析sql异常:【{}】", sql);
            } else {
                LOGGER.error("====>解析sql获取结果列异常,异常sql:【{}】", sql);
                throw new DatabaseEncryptException(DatabaseEncryptExceptionEnum.PARSE_SQL, sql);
            }
        }
        mappingMap.put(id, objectRelationalMappings);
        return objectRelationalMappings;
    }

    /**
     * SQL中表名与表别名的映射关系
     **/
    static class TableInfo {
        private final String tableName;

        private final String aliasTableName;

        public TableInfo(String tableName, String aliasTableName) {
            this.tableName = tableName;
            this.aliasTableName = aliasTableName;
        }

        public String getTableName() {
            return tableName;
        }

        public String getAliasTableName() {
            return aliasTableName;
        }
    }
}
