package com.digiwin.dap.middleware.dict.service.impl;

import com.digiwin.dap.middleware.dict.domain.DictDTO;
import com.digiwin.dap.middleware.dict.entity.Dict;
import com.digiwin.dap.middleware.dict.mapper.DictDataMapper;
import com.digiwin.dap.middleware.dict.mapper.DictMapper;
import com.digiwin.dap.middleware.dict.service.DictService;
import com.digiwin.dap.middleware.domain.CommonErrorCode;
import com.digiwin.dap.middleware.util.AssertUtils;
import com.digiwin.dap.middleware.util.EntityUtils;
import com.digiwin.dap.middleware.util.SnowFlake;
import com.github.pagehelper.PageSerializable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

/**
 * 字典类型Service业务层处理
 *
 * @author chenzhuang
 * @date 2022-09-05
 */
@Service
public class DictServiceImpl implements DictService {
    @Autowired
    private DictMapper dictMapper;
    @Autowired
    private DictDataMapper dictDataMapper;

    @Override
    public Dict selectDictBySid(Long sid) {
        return dictMapper.selectDict(new DictDTO(sid, null));
    }

    @Override
    public Dict selectDictById(String id) {
        return dictMapper.selectDict(new DictDTO(null, id));
    }

    @Override
    public Dict selectDict(DictDTO dto) {
        return dictMapper.selectDict(dto);
    }

    /**
     * 查询字典类型列表
     *
     * @param dto 字典类型
     * @return 字典类型
     */
    @Override
    public List<Dict> selectDictList(DictDTO dto) {
        return dictMapper.selectDictList(Optional.ofNullable(dto).orElse(new DictDTO()));
    }

    @Override
    public PageSerializable<Dict> selectDictList(DictDTO dto, int pageNum, int pageSize, String orderBy) {
        return PageSerializable.of(dictMapper.selectDictList(Optional.ofNullable(dto).orElse(new DictDTO()), pageNum, pageSize, orderBy));
    }

    /**
     * 新增字典类型
     *
     * @param dict 字典类型
     * @return 结果
     */
    @Override
    public int insertDict(Dict dict) {
        Dict exists = this.selectDictById(dict.getId());
        AssertUtils.valueIsNull(exists, dict.getId());
        EntityUtils.setCreateFields(dict);
        dict.setSid(SnowFlake.getInstance().newId());
        return dictMapper.insertDict(dict);
    }

    /**
     * 修改字典类型
     *
     * @param dict 字典类型
     * @return 结果
     */
    @Override
    public int updateDict(Dict dict) {
        Dict exists = this.selectDictById(dict.getId());
        AssertUtils.isTrue(exists == null || dict.getSid() == exists.getSid(), CommonErrorCode.VALUE_EXISTS, new Object[]{dict.getId()});
        EntityUtils.setModifyFields(dict);
        return dictMapper.updateDict(dict);
    }

    /**
     * 删除字典类型对象
     *
     * @param sids 需要删除的数据ID
     * @return 结果
     */
    @Override
    public int deleteDictByIds(Collection<Long> sids) {
        return dictMapper.deleteDictBySids(sids);
    }

    /**
     * 删除字典类型信息
     *
     * @param sid 字典类型ID
     * @return 结果
     */
    @Transactional
    @Override
    public int deleteDictBySid(Long sid) {
        Dict dict = this.selectDictBySid(sid);
        Optional.ofNullable(dict).ifPresent(x-> dictDataMapper.deleteDictDataByDictId(x.getId()));
        return dictMapper.deleteDictBySids(Collections.singletonList(sid));
    }
}
