package com.digiwin.dap.middleware.dict.service.impl;

import com.digiwin.dap.middleware.dict.domain.DictDataDTO;
import com.digiwin.dap.middleware.dict.entity.Dict;
import com.digiwin.dap.middleware.dict.entity.DictData;
import com.digiwin.dap.middleware.dict.mapper.DictDataMapper;
import com.digiwin.dap.middleware.dict.service.DictDataService;
import com.digiwin.dap.middleware.dict.service.DictService;
import com.digiwin.dap.middleware.domain.CommonErrorCode;
import com.digiwin.dap.middleware.util.AssertUtils;
import com.digiwin.dap.middleware.util.EntityUtils;
import com.digiwin.dap.middleware.util.SnowFlake;
import com.github.pagehelper.PageSerializable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

/**
 * 字典数据Service业务层处理
 *
 * @author chenzhuang
 * @date 2022-09-05
 */
@Service
public class DictDataServiceImpl implements DictDataService {
    @Autowired
    private DictDataMapper dictDataMapper;
    @Autowired
    private DictService dictService;

    /**
     * 查询字典数据
     *
     * @param sid 字典数据ID
     * @return 字典数据
     */
    @Override
    public DictData selectDictDataBySid(Long sid) {
        return dictDataMapper.selectDictData(new DictDataDTO(sid));
    }

    /**
     * 查询字典数据
     *
     * @param dto 字典数据ID
     * @return 字典数据
     */
    @Override
    public DictData selectDictData(DictDataDTO dto) {
        return dictDataMapper.selectDictData(dto);
    }

    /**
     * 查询字典数据列表
     *
     * @param dto 字典数据
     * @return 字典数据
     */
    @Override
    public List<DictData> selectDictDataList(DictDataDTO dto) {
        return dictDataMapper.selectDictDataList(Optional.ofNullable(dto).orElse(new DictDataDTO()));
    }

    @Override
    public PageSerializable<DictData> selectDictDataList(DictDataDTO dto, Integer pageNum, Integer pageSize, String orderBy) {
        return PageSerializable.of(dictDataMapper.selectDictDataList(Optional.ofNullable(dto).orElse(new DictDataDTO()), pageNum, pageSize, Optional.ofNullable(orderBy).orElse("dd.dict_sort")));
    }

    /**
     * 新增字典数据
     *
     * @param dictData 字典数据
     * @return 结果
     */
    @Override
    public int insertDictData(DictData dictData) {
        Dict dict = dictService.selectDictById(dictData.getDictId());
        AssertUtils.valueNotNull(dict, dictData.getDictId());
        DictData exists = dictDataMapper.selectDictData(new DictDataDTO(dictData.getDictId(), dictData.getDictKey()));
        AssertUtils.valueIsNull(exists, String.format("%s[%s]", dict.getName(), dictData.getDictKey()));
        EntityUtils.setCreateFields(dictData);
        dictData.setSid(SnowFlake.getInstance().newId());
        return dictDataMapper.insertDictData(dictData);
    }

    /**
     * 修改字典数据
     *
     * @param dictData 字典数据
     * @return 结果
     */
    @Override
    public int updateDictData(DictData dictData) {
        Dict dict = dictService.selectDictById(dictData.getDictId());
        AssertUtils.valueNotNull(dict, dictData.getDictId());
        DictData exists = dictDataMapper.selectDictData(new DictDataDTO(dictData.getDictId(), dictData.getDictKey()));
        AssertUtils.isTrue(exists == null || dictData.getSid() == exists.getSid(), CommonErrorCode.VALUE_EXISTS, new Object[]{String.format("DictId[%s],DictKey[%s]", dictData.getDictId(), dictData.getDictKey())});
        EntityUtils.setModifyFields(dictData);
        return dictDataMapper.updateDictData(dictData);
    }

    /**
     * 删除字典数据对象
     *
     * @param sids 需要删除的数据ID
     * @return 结果
     */
    @Override
    public int deleteDictDataBySids(Collection<Long> sids) {
        return dictDataMapper.deleteDictDataBySids(sids);
    }

    /**
     * 删除字典数据信息
     *
     * @param sid 字典数据ID
     * @return 结果
     */
    @Override
    public int deleteDictDataBySid(Long sid) {
        return dictDataMapper.deleteDictDataBySids(Collections.singletonList(sid));
    }

}
