package com.digiwin.dap.middle.encrypt.domain;

import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.LocalDateTimeUtil;
import com.digiwin.dap.middle.encrypt.contstant.EncryptConstants;
import com.digiwin.dap.middleware.domain.CommonErrorCode;
import com.digiwin.dap.middleware.exception.BusinessException;
import com.digiwin.dap.middleware.util.JsonUtils;
import org.springframework.util.ObjectUtils;

import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;
import java.time.format.DateTimeParseException;
import java.util.StringJoiner;

/**
 * 签名信息
 *
 * @author ChenZhuang
 * @date 2024-01-17 14:38:00
 */
public class DapSignInfo {
    private String nonce;
    private String timestamp;
    private String sign;

    public static DapSignInfo get(String json) {
        try {
            String decode = URLDecoder.decode(json, StandardCharsets.UTF_8.name());
            return JsonUtils.createObjectMapper().readValue(decode, DapSignInfo.class);
        } catch (Exception e) {
            throw new BusinessException(CommonErrorCode.SIGN_HEADER_ARG_PARSING_FAILED, new Object[]{json});
        }
    }

    /**
     * 校验参数值规范
     *
     * @param o the o
     */
    public static void verify(DapSignInfo o) {
        if (o == null) {
            throw new BusinessException(CommonErrorCode.SIGN_HEADER_ARG_EMPTY);
        }
        if (ObjectUtils.isEmpty(o.getNonce())) {
            throw new BusinessException(CommonErrorCode.SIGN_NONCE_EMPTY);
        }
        if (o.getNonce().length() < EncryptConstants.NONCE_MIN_LENGTH || o.getNonce().length() > EncryptConstants.NONCE_MAX_LENGTH) {
            throw new BusinessException(CommonErrorCode.SIGN_NONCE_LENGTH_ERROR, new Object[]{o.getNonce()});
        }

        if (ObjectUtils.isEmpty(o.getTimestamp())) {
            throw new BusinessException(CommonErrorCode.SIGN_TIMESTAMP_EMPTY);
        }
        try {
            LocalDateTime dateTime = LocalDateTimeUtil.parse(o.getTimestamp(), DatePattern.PURE_DATETIME_PATTERN);
        } catch (DateTimeParseException e) {
            throw new BusinessException(CommonErrorCode.SIGN_TIMESTAMP_FORMAT_ERROR, new Object[]{o.getTimestamp()});
        }

        if (ObjectUtils.isEmpty(o.getSign())) {
            throw new BusinessException(CommonErrorCode.SIGNATURE_SIGN_EMPTY);
        }
    }

    public String getNonce() {
        return nonce;
    }

    public void setNonce(String nonce) {
        this.nonce = nonce;
    }

    public String getTimestamp() {
        return timestamp;
    }

    public void setTimestamp(String timestamp) {
        this.timestamp = timestamp;
    }

    public String getSign() {
        return sign;
    }

    public void setSign(String sign) {
        this.sign = sign;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", DapSignInfo.class.getSimpleName() + "[", "]")
                .add("nonce='" + nonce + "'")
                .add("timestamp='" + timestamp + "'")
                .add("sign='" + sign + "'")
                .toString();
    }
}
