package com.digiwin.dap.middle.gateway.service.track.http;

import com.digiwin.dap.middleware.auth.AuthoredSys;
import com.digiwin.dap.middleware.auth.AuthoredUser;
import com.digiwin.dap.middleware.util.UserUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.InitializingBean;

import java.net.URI;
import java.time.Instant;
import java.util.Map;

public class HttpTrack implements InitializingBean {

    private final Instant timestamp;
    private final Request request;

    private volatile Response response;
    private volatile Long timeTaken;

    private volatile AuthoredUser user;
    private volatile AuthoredSys sys;

    public HttpTrack(TraceableRequest request) {
        this.request = new Request(request);
        this.timestamp = Instant.now();
    }

    public Instant getTimestamp() {
        return timestamp;
    }

    public Request getRequest() {
        return request;
    }

    public Response getResponse() {
        return response;
    }

    public void setResponse(Response response) {
        this.response = response;
    }

    public Long getTimeTaken() {
        return timeTaken;
    }

    public void setTimeTaken(Long timeTaken) {
        this.timeTaken = timeTaken;
    }

    public AuthoredUser getUser() {
        return user;
    }

    public void setUser(AuthoredUser user) {
        this.user = user;
    }

    public AuthoredSys getSys() {
        return sys;
    }

    public void setSys(AuthoredSys sys) {
        this.sys = sys;
    }

    @Override
    public void afterPropertiesSet() {
        try {
            this.timeTaken = System.currentTimeMillis() - this.timestamp.toEpochMilli();

            AuthoredUser authoredUser = UserUtils.getAuthoredUser();
            if (authoredUser != null) {
                this.user = new AuthoredUser();
                BeanUtils.copyProperties(authoredUser, user);
            }

            AuthoredSys authoredSys = UserUtils.getAuthoredSys();
            if (authoredSys != null) {
                this.sys = new AuthoredSys();
                BeanUtils.copyProperties(authoredSys, sys);
            }
        } catch (Exception ignored) {
        }
    }

    /**
     * Trace of an HTTP request.
     */
    public static final class Request {

        private final String method;

        private final URI uri;

        private final Map<String, String> headers;

        private final String remoteAddress;

        private Request(TraceableRequest request) {
            this.method = request.getMethod();
            this.uri = request.getUri();
            this.headers = request.getHeaders();
            this.remoteAddress = request.getRemoteAddress();
        }

        public String getMethod() {
            return method;
        }

        public URI getUri() {
            return uri;
        }

        public Map<String, String> getHeaders() {
            return headers;
        }

        public String getRemoteAddress() {
            return remoteAddress;
        }
    }

    /**
     * Trace of an HTTP response.
     */
    public static final class Response {

        private final int status;

        private final Map<String, String> headers;

        public Response(TraceableResponse response, int status) {
            this.status = status;
            this.headers = response.getHeaders();
        }

        public int getStatus() {
            return this.status;
        }

        public Map<String, String> getHeaders() {
            return headers;
        }
    }
}
