package com.digiwin.dap.middle.gateway.service.log;

import com.digiwin.dap.middleware.constant.DapHttpHeaders;
import com.digiwin.dap.middleware.constant.GlobalConstants;
import com.digiwin.dap.middleware.lmc.LMC;
import com.digiwin.dap.middleware.lmc.request.SaveEventLog;
import com.digiwin.dap.middleware.util.UserUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpRequest;
import org.springframework.http.client.ClientHttpRequestExecution;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.ClientHttpResponse;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.LinkedHashMap;
import java.util.Map;

public class DapLoggingInterceptor implements ClientHttpRequestInterceptor {

    private static final Logger logger = LoggerFactory.getLogger(DapLoggingInterceptor.class);

    private final String appName;
    private final LMC lmcClient;

    public DapLoggingInterceptor(String appName, LMC lmcClient) {
        this.appName = appName;
        this.lmcClient = lmcClient;
    }

    @Override
    public ClientHttpResponse intercept(HttpRequest request, byte[] body, ClientHttpRequestExecution execution) throws IOException {
        String errorMsg = null;
        try {
            ClientHttpResponse response = execution.execute(request, body);
            if (response.getStatusCode().isError()) {
                errorMsg = response.getStatusCode().toString();
            }
            return response;
        } catch (IOException e) {
            errorMsg = e.getMessage();
            throw e;
        } finally {
            if (errorMsg != null) {
                logRequestEx(request, body, errorMsg);
            }
        }
    }

    public void logRequestEx(HttpRequest request, byte[] body, String errorMsg) {
        try {
            SaveEventLog eventLog = new SaveEventLog();
            // 基础信息
            eventLog.setAppId(GlobalConstants.COMMON_APP);
            eventLog.setEventSource(appName);
            eventLog.setEventType(99);
            eventLog.setEventId("INTERNAL_SERVICE_CALL");
            eventLog.setEventName("内部服务调用");
            eventLog.setUserId(UserUtils.getUserId());
            eventLog.setUserName(UserUtils.getUserName());
            eventLog.setTenantId(UserUtils.getTenantId());
            eventLog.setTenantName(UserUtils.getTenantName());
            eventLog.setSysId(UserUtils.getSysId());
            // 异常信息
            eventLog.setStatus(1);
            eventLog.setErrorMsg(errorMsg);
            // 请求信息处理
            eventLog.setMethod(request.getMethod().name());
            eventLog.setRequestUrl(request.getURI().getPath());
            eventLog.setRequestParam(request.getURI().getQuery());
            // 请求参数
            Map<String, Object> contents = new LinkedHashMap<>();
            contents.put("uri", request.getURI());
            contents.put("method", request.getMethod());
            contents.put("headers", extractHeaders(request));
            contents.put("requestBody", new String(body, StandardCharsets.UTF_8));
            eventLog.setContent(contents);

            lmcClient.saveEventLog(eventLog);
        } catch (Exception ex) {
            logger.error("记录内部调用异常", ex);
        }
    }

    private Map<String, String> extractHeaders(HttpRequest request) {
        Map<String, String> headers = new LinkedHashMap<>();
        request.getHeaders().forEach((key, value) -> {
            if (DapHttpHeaders.contains(key)) {
                headers.put(key, value.get(0));
            }
        });
        return headers;
    }
}
