package com.digiwin.dap.middle.gateway.service.stats.filter;


import com.digiwin.dap.middleware.auth.AppAuthContext;
import com.digiwin.dap.middleware.auth.AppAuthContextHolder;
import com.digiwin.dap.middleware.auth.domain.RequestInfo;
import com.digiwin.dap.middleware.cache.RedisUtils;
import com.digiwin.dap.middleware.domain.FilterOrderEnum;
import org.springframework.core.Ordered;
import org.springframework.web.filter.OncePerRequestFilter;

import jakarta.servlet.FilterChain;
import jakarta.servlet.ServletException;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.concurrent.TimeUnit;

/**
 * http请求记录
 *
 * @author fobgochod
 */
public class HttpStatsFilter extends OncePerRequestFilter implements Ordered {

    private static final String EMPTY_SYS = "empty";
    private static final String STATS_SYS_PREFIX = ":stats:sys:";
    private final String appName;

    public HttpStatsFilter(String appName) {
        this.appName = appName;
    }

    /**
     * hash添加一个过期时间，防止垃圾数据
     *
     * @param key hash key
     */
    private static void setExpire(String key) {
        Long expire = RedisUtils.getExpire(key);
        if (expire == null || expire == -1L) {
            RedisUtils.expire(key, 7, TimeUnit.DAYS);
        }
    }

    @Override
    public int getOrder() {
        return FilterOrderEnum.API_STATS.order();
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        final AppAuthContext context = AppAuthContextHolder.getContext();
        final RequestInfo requestInfo = context.getRequestInfo();
        try {
            requestInfo.setM(request.getMethod());
            requestInfo.setP(request.getRequestURI());
            filterChain.doFilter(request, response);
        } finally {
            String sysId = context.getAuthoredSys() == null ? EMPTY_SYS : context.getAuthoredSys().getId();
            String today = LocalDate.now().format(DateTimeFormatter.BASIC_ISO_DATE);
            // appName:stats:sys:YYYYMMDD:status:method-path
            String key = appName + STATS_SYS_PREFIX + today + ":" + response.getStatus() + ":" + requestInfo.getM() + "-" + requestInfo.getP();
            RedisUtils.opsForHash().increment(key, sysId, 1);
        }
    }
}
