package com.digiwin.dap.middle.ha;

import com.digiwin.dap.middle.ha.condition.DeployAreaCondition;
import com.digiwin.dap.middle.ha.config.HealthRedisTemplate;
import com.digiwin.dap.middle.ha.config.RedisHealthProperties;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Conditional;
import org.springframework.data.redis.connection.RedisPassword;
import org.springframework.data.redis.connection.RedisStandaloneConfiguration;
import org.springframework.data.redis.connection.jedis.JedisClientConfiguration;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import redis.clients.jedis.JedisPoolConfig;

import java.time.Duration;

/**
 * 专门建立一个healthRedisTemplate 用来监控常用地址心跳
 * 异常时切换其他Redis主机（微软正式区、开发区启动）
 *
 * @author fobgochod
 * @date 2021/2/1
 */
@AutoConfiguration
@ComponentScan(basePackages = "com.digiwin.dap.middle.ha")
@ConditionalOnProperty(name = "dap.middleware.ha.redis.host")
@Conditional(DeployAreaCondition.class)
public class RedisHealthConfiguration {

    @Autowired
    private RedisHealthProperties redisHealthProperties;

    @Bean
    public HealthRedisTemplate healthRedisTemplate() {
        return new HealthRedisTemplate(createJedisConnectionFactory());
    }

    private JedisConnectionFactory createJedisConnectionFactory() {
        JedisClientConfiguration clientConfiguration = getJedisClientConfiguration();
        JedisConnectionFactory jedisConnectionFactory = new JedisConnectionFactory(getStandaloneConfig(), clientConfiguration);
        jedisConnectionFactory.afterPropertiesSet();
        return jedisConnectionFactory;
    }

    private JedisClientConfiguration getJedisClientConfiguration() {
        JedisClientConfiguration.JedisClientConfigurationBuilder builder = applyProperties(JedisClientConfiguration.builder());
        builder.usePooling().poolConfig(jedisPoolConfig());
        return builder.build();
    }

    private RedisStandaloneConfiguration getStandaloneConfig() {
        RedisStandaloneConfiguration config = new RedisStandaloneConfiguration();
        config.setHostName(redisHealthProperties.getRedisHost());
        config.setPort(redisHealthProperties.getRedisPort());
        config.setPassword(RedisPassword.of(redisHealthProperties.getRedisPassword()));
        config.setDatabase(redisHealthProperties.getRedisDb());
        return config;
    }

    private JedisClientConfiguration.JedisClientConfigurationBuilder applyProperties(JedisClientConfiguration.JedisClientConfigurationBuilder builder) {
        Duration timeout = Duration.ofSeconds(4);
        builder.readTimeout(timeout).connectTimeout(timeout);
        return builder;
    }

    private JedisPoolConfig jedisPoolConfig() {
        JedisPoolConfig config = new JedisPoolConfig();
        config.setMaxTotal(1);
        config.setMaxIdle(1);
        config.setMinIdle(1);
        config.setMaxWait(Duration.ofSeconds(5));
        return config;
    }
}

