package com.digiwin.dap.middle.license.utils;

import com.sun.jna.Library;
import com.sun.jna.Native;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;


public class NativeUtils {

    public static final String NATIVE_FOLDER_PATH_PREFIX = "native";
    private static File temporaryDir;

    private NativeUtils() {
    }

    public static synchronized void load(String libName) {
        try {
            String libExtension = SystemArch.getExt();
            String libFilename = libName + libExtension;
            File tempFile = createTempFile(libFilename);
            try {
                System.load(tempFile.getAbsolutePath());
            } finally {
                tempFile.deleteOnExit();
            }
        } catch (IOException e) {
            throw new RuntimeException("Failed to load library: " + libName, e);
        }
    }

    public static synchronized void load0(String libFilename) {
        try {
            File tempFile = createTempFile(libFilename);
            try {
                System.load(tempFile.getAbsolutePath());
            } finally {
                tempFile.deleteOnExit();
            }
        } catch (IOException e) {
            throw new RuntimeException("Failed to load library: " + libFilename, e);
        }
    }


    public static <T extends Library> T loadLibrary(String libName, Class<T> interfaceClass) {
        try {
            String libExtension = SystemArch.getExt();
            String libFilename = libName + libExtension;
            File tempFile = createTempFile(libFilename);
            try {
                return Native.load(tempFile.getAbsolutePath(), interfaceClass);
            } finally {
                tempFile.deleteOnExit();
            }
        } catch (IOException e) {
            throw new RuntimeException("Failed to load library: " + libName, e);
        }
    }


    private static File createTempDirectory() throws IOException {
        String tempDir = "target/" + NATIVE_FOLDER_PATH_PREFIX;

        File temp = new File(tempDir);
        if (temp.exists()) {
            return temp;
        }
        if (temp.mkdirs()) {
            temp.setWritable(true, false);
            temp.setExecutable(true, false);
            temp.setReadable(true, false);
            return temp;
        } else {
            throw new IOException("Failed to create temp directory " + temp.getName());
        }
    }

    private static File createTempFile(String libName) throws IOException {
        String resourcePath = "/" + SystemArch.getLibraryFold() + "/" + libName;

        if (temporaryDir == null) {
            temporaryDir = createTempDirectory();
            temporaryDir.deleteOnExit();
        }
        File tempFile = new File(temporaryDir, libName);
        try (InputStream is = NativeUtils.class.getResourceAsStream(resourcePath)) {
            Files.copy(is, tempFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
            tempFile.setWritable(true, false);
            tempFile.setExecutable(true, false);
            tempFile.setReadable(true, false);
        } catch (IOException e) {
            tempFile.delete();
            throw e;
        } catch (NullPointerException e) {
            tempFile.delete();
            throw new FileNotFoundException("File " + resourcePath + " was not found inside JAR.");
        }
        return tempFile;
    }

}
