package com.digiwin.dap.middle.license.utils;

import com.digiwin.dap.middle.license.bean.ShellInfo;
import com.digiwin.dap.middleware.commons.util.StrUtils;
import com.jcraft.jsch.ChannelExec;
import com.jcraft.jsch.JSch;
import com.jcraft.jsch.Session;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.util.ArrayList;
import java.util.List;

/**
 * 连接SSH
 * 执行命令Linux命令
 *
 * @author fobgochod
 */
public class SshUtils {

    private static final Logger logger = LoggerFactory.getLogger(SshUtils.class);

    public static Session getSession(ShellInfo shellInfo) {
        try {
            JSch jSch = new JSch();
            Session session = jSch.getSession(shellInfo.getUsername(), shellInfo.getHost(), shellInfo.getPort());
            if (StrUtils.isNotEmpty(shellInfo.getIdentity())) {
                if (isPrivateKeyFile(shellInfo.getIdentity())) {
                    jSch.addIdentity(shellInfo.getIdentity(), shellInfo.getPassphrase());
                } else if (isPrivateKeyString(shellInfo.getIdentity())) {
                    jSch.addIdentity("id_rsa", shellInfo.getIdentity().getBytes(), null, null);
                }
            } else if (StrUtils.isNotEmpty(shellInfo.getPassword())) {
                session.setPassword(shellInfo.getPassword());
            }
            // 禁用主机密钥检查
            session.setConfig("StrictHostKeyChecking", "no");
            return session;
        } catch (Exception e) {
            logger.error("JSch create session error:", e);
            throw new RuntimeException(e);
        }
    }


    public static String execOne(Session session, String command) {
        List<String> results = exec(session, command);
        return results.isEmpty() ? null : results.get(0);
    }

    /**
     * 执行命令
     *
     * @param session 会话
     * @param command 执行命令
     * @return 执行结果
     */
    public static List<String> exec(Session session, String command) {
        logger.debug(">> {}", command);
        List<String> resultLines = new ArrayList<>();
        ChannelExec channel = null;
        try {
            channel = (ChannelExec) session.openChannel("exec");
            channel.setCommand(command);
            channel.setPty(true);
            InputStream is = channel.getInputStream();
            channel.connect(ShellInfo.CONNECT_TIMEOUT);
            try {
                BufferedReader inputReader = new BufferedReader(new InputStreamReader(is));
                String inputLine;
                while ((inputLine = inputReader.readLine()) != null) {
                    resultLines.add(inputLine);
                }
            } finally {
                if (is != null) {
                    try {
                        is.close();
                    } catch (Exception e) {
                        logger.error("JSch inputStream close error:", e);
                    }
                }
            }
        } catch (Exception e) {
            logger.error("IOException:", e);
        } finally {
            if (channel != null) {
                try {
                    channel.disconnect();
                } catch (Exception e) {
                    logger.error("JSch channel disconnect error:", e);
                }
            }
        }
        return resultLines;
    }

    public static void connect(Session session) {
        if (session == null) return;
        try {
            session.connect(ShellInfo.CONNECT_TIMEOUT);
            if (session.isConnected()) {
                logger.info("Host({}) connected.", session.getHost());
                return;
            }
        } catch (Exception e) {
            logger.error("JSch create session error:", e);
        }
        throw new RuntimeException(String.format("Host(%s) connection failed.", session.getHost()));
    }

    public static void close(Session session) {
        if (session == null) return;
        try {
            session.disconnect();
        } catch (Exception e) {
            logger.error("JSch channel disconnect error:", e);
        }
    }


    // 判断是否为私钥文件路径
    private static boolean isPrivateKeyFile(String input) throws IOException {
        File file = new File(input);
        if (file.exists() && file.isFile() && file.canRead()) {
            try (BufferedReader reader = new BufferedReader(new FileReader(file))) {
                String line;
                StringBuilder content = new StringBuilder();
                while ((line = reader.readLine()) != null) {
                    content.append(line).append("\n");
                }
                return isPrivateKeyString(content.toString());
            }
        }
        return false;
    }

    // 判断是否为私钥字符串
    private static boolean isPrivateKeyString(String input) {
        String trimmedInput = input.trim();
        return (trimmedInput.startsWith("-----BEGIN PRIVATE KEY-----") &&
                trimmedInput.endsWith("-----END PRIVATE KEY-----")) ||
                (trimmedInput.startsWith("-----BEGIN RSA PRIVATE KEY-----") &&
                        trimmedInput.endsWith("-----END RSA PRIVATE KEY-----")) ||
                (trimmedInput.startsWith("-----BEGIN OPENSSH PRIVATE KEY-----") &&
                        trimmedInput.endsWith("-----END OPENSSH PRIVATE KEY-----"));
    }
}
