package com.digiwin.dap.middle.license.utils;

import com.sun.jna.Platform;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SystemArch {

    private static final Logger logger = LoggerFactory.getLogger(SystemArch.class);

    private static final String EXTERNAL_WIN = ".dll";
    private static final String EXTERNAL_LINUX = ".so";
    private static final String EXTERNAL_MAC = ".dylib";

    private static final String NATIVE_PATH = "native";
    private static final String ARCH_WINDOWS = "windows";
    private static final String PREFIX_64 = "win64";
    private static final String PREFIX_32 = "win32";

    private static final String ARCH_LINUX = "linux";
    private static final String PREFIX_AMD = "x86";
    private static final String PREFIX_ARM = "arm";
    private static final String PREFIX_LONG = "loongarch";
    private static final String PREFIX_MIPS = "mips";

    private static final String ARCH_MAC = "mac";

    private SystemArch() {
    }

    public static String getExt() {
        return Platform.isWindows() ? EXTERNAL_WIN : (Platform.isMac() ? EXTERNAL_MAC : EXTERNAL_LINUX);
    }

    public static String getLibraryFold() {
        String osName = System.getProperty("os.name").toLowerCase();
        String arch = System.getProperty("os.arch").toLowerCase();
        logger.info("Operating system info: {} {}", osName, arch);

        StringBuilder dirPath = new StringBuilder();
        dirPath.append(NATIVE_PATH);
        if (Platform.isLinux()) {
            dirPath.append("/" + ARCH_LINUX);
            dirPath.append("/").append(getLinux(arch));
        } else if (Platform.isWindows()) {
            dirPath.append("/" + ARCH_WINDOWS);
            dirPath.append("/").append(Platform.is64Bit() ? PREFIX_64 : PREFIX_32);
        } else if (Platform.isMac()) {
            dirPath.append("/" + ARCH_MAC);
        }
        logger.info("Dependency library file placement location: {}", dirPath);
        return dirPath.toString();
    }

    private static String getLinux(String arch) {
        if ("x86_64".startsWith(arch) || "amd64".startsWith(arch)) {
            return PREFIX_AMD;
        }
        if (Platform.isARM()) {
            return PREFIX_ARM;
        }
        if (Platform.isLoongArch()) {
            return PREFIX_LONG;
        }
        if (Platform.isMIPS()) {
            return PREFIX_MIPS;
        }
        return "";
    }
}
