package com.digiwin.dap.middleware.mybatis;

import com.fasterxml.jackson.annotation.JsonIgnore;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import static java.lang.annotation.ElementType.*;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

/**
 * 自动解密注解，用于加在实体类和对应字段上
 * 注意：原始字段最好也是字符串类型，因为只有字符串类型才会默认赋值，否则需要自己去再次赋值
 * 注意一般需要吧临时加密字段，去除序列化添加{@link JsonIgnore}
 * 同时如果mybatis的mapper方法出参用的是jpa的原始实体类，还需要添加javax.persistence.Transient注解忽略存储该字段<p>
 * example：<p>
 *     <pre>
 *     &#064;AutoDecrypt(originalFieldName = "code")
 *     &#064;JsonIgnore
 *     &#064;Transient
 *     private String codeToDecrypt;
 *     </pre>
 *
 */
@Target({METHOD, FIELD, TYPE}) @Retention(RUNTIME)
public @interface AutoDecrypt {
    /**
     * 仅用于注解在字段的情况，也可以缺省，字段名满足规则  ：   <原始字段名>ToDecrypt
     * example：<p>
     *     <pre>
     *     &#064;Column(name = "code", columnDefinition = "VARCHAR(50) NOT NULL COMMENT 'code'")
     *     private String code;
     *
     *     &#064;AutoDecrypt
     *     &#064;JsonIgnore
     *     &#064;Transient
     *     private String codeToDecrypt;
     *     </pre>
     */
    String originalFieldName() default "";

    /**
     * 如果明文字段也是字符串,可以直接注解加在明文字段上,并且用本属性标识{@code  true}<p>
     * example:<p>
     *     <pre>
     *     &#064;Column(name = "code", columnDefinition = "VARCHAR(50) NOT NULL COMMENT 'code'")
     *     &#064;AutoDecrypt(selfLoad = true)
     *     private String code;
     *     </pre>
     */
    boolean selfLoad() default false;
}
