package com.digiwin.dap.middleware.mybatis;

import cn.hutool.core.util.ReflectUtil;
import com.digiwin.dap.middleware.mybatis.typeHandler.DecryptedFieldHandler;
import com.digiwin.dap.middleware.mybatis.typeHandler.IntegerEnAndDecryptTypeHandler;
import com.digiwin.dap.middleware.mybatis.typeHandler.StringEnAndDecryptTypeHandler;
import com.digiwin.dap.middleware.util.SecureUtils;
import com.digiwin.dap.middleware.util.StringUtils;
import org.apache.ibatis.cache.CacheKey;
import org.apache.ibatis.executor.Executor;
import org.apache.ibatis.mapping.BoundSql;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.plugin.Interceptor;
import org.apache.ibatis.plugin.Intercepts;
import org.apache.ibatis.plugin.Invocation;
import org.apache.ibatis.plugin.Signature;
import org.apache.ibatis.session.ResultHandler;
import org.apache.ibatis.session.RowBounds;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Field;
import java.util.*;

/**
 * 全局解密拦截器，适用所有类型字段自动解密
 * 如果无须使用,就别引入spring容器
 *
 * @author blockWilling
 * @date 2024/4/11 10:38
 * @mail kangjin@digiwin.com
 */
@Intercepts({
        @Signature(type = Executor.class, method = "query", args = {MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class}),
        @Signature(type = Executor.class, method = "query", args = {MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class, CacheKey.class, BoundSql.class}),
})
public class DecryptInterceptor implements Interceptor {
    private static final int PARAM_COUNT = 4;
    private static final String TMP_FIELD_SUFFIX = "ToDecrypt";
    private static final Logger logger = LoggerFactory.getLogger(DecryptInterceptor.class);
    private Set<DecryptedFieldHandler<?>> set = new HashSet<>();

    public DecryptInterceptor() {
        set.add(new IntegerEnAndDecryptTypeHandler());
        set.add(new StringEnAndDecryptTypeHandler());
    }

    public Set<DecryptedFieldHandler<?>> getDecryptedFieldHandlers() {
        return set;
    }

    /**
     * 自定义{@link DecryptedFieldHandler}集合
     *
     * <pre>
     * &#064;Bean public DecryptInterceptor decryptInterceptor() {
     * return new DecryptInterceptor().customizeDecryptedFieldHandler(...);
     * }
     * </>
     * @param set
     */

    public DecryptInterceptor customizeDecryptedFieldHandlers(Set<DecryptedFieldHandler<?>> set) {
        this.set = set;
        return this;
    }

    @Override
    public Object intercept(Invocation invocation) throws Throwable {
        Object[] args = invocation.getArgs();
        MappedStatement ms = (MappedStatement) args[0];
        Object parameter = args[1];
        RowBounds rowBounds = (RowBounds) args[2];
        ResultHandler resultHandler = (ResultHandler) args[3];
        Executor executor = (Executor) invocation.getTarget();
        CacheKey cacheKey;
        BoundSql boundSql;
        //由于逻辑关系，只会进入一次
        if (args.length == PARAM_COUNT) {
            //4 个参数时
            boundSql = ms.getBoundSql(parameter);
            cacheKey = executor.createCacheKey(ms, parameter, rowBounds, boundSql);
        } else {
            //6 个参数时
            cacheKey = (CacheKey) args[4];
            boundSql = (BoundSql) args[5];
        }
        List<Object> ret = executor.query(ms, parameter, rowBounds, resultHandler, cacheKey, boundSql);

        if (ret==null||ret.isEmpty()) {
            return ret;
        }
        try {
            Object first = ret.stream().filter(Objects::nonNull).findFirst().get();
            Class<?> firstClass = first.getClass();
            //是否需要解密
            AutoDecrypt annotation = firstClass.getAnnotation(AutoDecrypt.class);
            if (annotation == null) {
                return ret;
            }
            //遍历解密
            for (Object o : ret) {
                if (o == null) {
                    continue;
                }
                Field[] fields = ReflectUtil.getFields(o.getClass());
                for (Field field : fields) {
                    field.setAccessible(true);
                    AutoDecrypt annotationField = field.getAnnotation(AutoDecrypt.class);
                    if (annotationField == null) {
                        continue;
                    }
                    String originalColName = annotationField.originalFieldName();
                    boolean selfLoad = annotationField.selfLoad();
                    //必须使用字符串声明解密字段，这里是添加的待解密字段，作为临时载体
                    if (String.class.equals(field.getType())) {
                        String toDecrypt = (String) field.get(o);
                        Field originalField = null;
                        if (selfLoad) {
                            originalField = field;
                        } else if (originalColName==null||originalColName.isEmpty()) {
                            String name = field.getName();
                            int i = name.indexOf(TMP_FIELD_SUFFIX);
                            if (i == -1 || i == 0) {
                                logger.error("解密字段注解配置有误:{}", name);
                                continue;
                            }
                            originalColName = name.substring(0, i);
                        }
                        if (originalField == null) {
                            originalField = ReflectUtil.getField(o.getClass(), originalColName);
                        }
                        originalField.setAccessible(true);
                        String decrypt = null;
                        try {
                            decrypt = SecureUtils.aesDecrypt(toDecrypt);
                        } catch (Exception e) {
                            logger.error("【SecureUtils.aesDecrypt】{}", e.getMessage());
                            continue;
                        }
                        for (DecryptedFieldHandler<?> decryptedFieldHandler : set) {
                            if (decryptedFieldHandler.support(originalField)) {
                                originalField.set(o, decryptedFieldHandler.handle(decrypt));
                                break;
                            }
                        }

                    }
                }
            }
        } catch (Exception e) {
            logger.error("【DecryptInterceptor】", e);
        }
        return ret;
    }
}
