package com.digiwin.dap.middleware.mybatis.interceptor;

import com.digiwin.dap.middle.database.encrypt.desensitization.DesensitizationProcessorFactory;
import com.digiwin.dap.middle.database.encrypt.enums.DesensitizationMode;
import com.digiwin.dap.middle.database.encrypt.exception.DatabaseEncryptException;
import com.digiwin.dap.middle.database.encrypt.model.ResultSetMappingRelation;
import com.digiwin.dap.middle.database.encrypt.model.SensitiveWordDatabase;
import com.digiwin.dap.middle.database.encrypt.sensitive.word.SensitiveWordDatabaseRegistry;
import com.digiwin.dap.middle.database.encrypt.utils.SqlParserUtils;
import org.apache.ibatis.cache.CacheKey;
import org.apache.ibatis.executor.Executor;
import org.apache.ibatis.mapping.BoundSql;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.plugin.Interceptor;
import org.apache.ibatis.plugin.Intercepts;
import org.apache.ibatis.plugin.Invocation;
import org.apache.ibatis.plugin.Signature;
import org.apache.ibatis.session.ResultHandler;
import org.apache.ibatis.session.RowBounds;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;

import java.lang.reflect.InvocationTargetException;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author michael
 * <p>mybatis拦截器,拦截所有select语句对结果集进行解密,需要注意一下三点：
 * <p>1.selec语句中不可出现select *</p>
 * <p>2.xml中如果配置了resultMap不支持解密,推荐使用自定义TypeHandler</p>
 * <p>3.结果集列上进行了函数操作不支持解密</p>
 * </p>
 */
@Intercepts({@Signature(
        type = Executor.class,
        method = "query",
        args = {MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class}
), @Signature(
        type = Executor.class,
        method = "query",
        args = {MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class, CacheKey.class, BoundSql.class}
)})
public class DecryptResultInterceptor implements Interceptor {

    private final static Logger LOGGER = LoggerFactory.getLogger(DecryptResultInterceptor.class);

    private final DesensitizationProcessorFactory desensitizationProcessorFactory;

    private final SensitiveWordDatabaseRegistry sensitiveWordDatabaseRegistry;

    public DecryptResultInterceptor(DesensitizationProcessorFactory desensitizationProcessorFactory,
                                    SensitiveWordDatabaseRegistry sensitiveWordDatabaseRegistry) {
        this.desensitizationProcessorFactory = desensitizationProcessorFactory;
        this.sensitiveWordDatabaseRegistry = sensitiveWordDatabaseRegistry;
    }

    @Override
    public Object intercept(Invocation invocation) throws InvocationTargetException, IllegalAccessException {
        Object result = null;
        Map<String, DesensitizationMode> fieldAndDesensitizationModeMap;
        try {

            MappedStatement mappedStatement = (MappedStatement) invocation.getArgs()[0];
            if (mappedStatement.getResultMaps() != null && !mappedStatement.getResultMaps().isEmpty()
                    && mappedStatement.getResultMaps().get(0).getResultMappings() != null && !mappedStatement.getResultMaps().get(0).getResultMappings().isEmpty()) {
                LOGGER.warn("===>ResultMap建议使用TypeHandler实现解密.");
                return invocation.proceed();
            }
            result = invocation.proceed();
            if (Objects.nonNull(result)) {
                // 获取mybatis当前查询结果集中涉及的敏感字段
                fieldAndDesensitizationModeMap = getFieldAndDesensitizationModeMap(invocation);
                desensitizationProcessorFactory.revert("", result, fieldAndDesensitizationModeMap);
            }
        } catch (Exception e) {
            if (e instanceof DatabaseEncryptException) {
                LOGGER.error("===>解密mybatis查询结果集异常", e);
            } else {
                // 继续抛出异常以便调用者处理
                throw e;
            }
        }
        return result;
    }

    @Override
    public Object plugin(Object target) {
        return Interceptor.super.plugin(target);
    }

    @Override
    public void setProperties(Properties properties) {
        Interceptor.super.setProperties(properties);
    }

    private Map<String, DesensitizationMode> getFieldAndDesensitizationModeMap(Invocation invocation) {
        try {
            Object[] args = invocation.getArgs();
            MappedStatement mappedStatement = (MappedStatement) args[0];
            String id = mappedStatement.getId();
            String sql = mappedStatement.getBoundSql(args[1]).getSql();

            Map<String, DesensitizationMode> fieldAndDesensitizationModeMap = new HashMap<>(16);

            // 1.解析sql获取数据库表名、表字段、java对象属性名之间的映射关系
            List<ResultSetMappingRelation> resultSetMappingRelationList = SqlParserUtils.parseQuerySql(sql);
            if (CollectionUtils.isEmpty(resultSetMappingRelationList)) {
                return fieldAndDesensitizationModeMap;
            }

            // 2.扫描全部实体类，根据表名获取相应实体列以及其敏感字段
            Map<String, List<ResultSetMappingRelation>> columnInfoMap = resultSetMappingRelationList.stream().collect(Collectors.groupingBy(ResultSetMappingRelation::getTableName));
            for (Map.Entry<String, List<ResultSetMappingRelation>> entry : columnInfoMap.entrySet()) {
                List<SensitiveWordDatabase> sensitiveWordDatabaseList = sensitiveWordDatabaseRegistry.findSensitiveWordDatabase(entry.getKey());
                if (CollectionUtils.isEmpty(sensitiveWordDatabaseList)) {
                    continue;
                }
                for (ResultSetMappingRelation resultSetMappingRelation : entry.getValue()) {
                    SensitiveWordDatabase sensitiveWordDatabase =
                            sensitiveWordDatabaseList.stream().filter(x -> Objects.equals(x.getColumnName(), resultSetMappingRelation.getColumnName())).findFirst().orElse(null);
                    if (Objects.nonNull(sensitiveWordDatabase)) {
                        LOGGER.info("===>mybatis中【{}】发法返回结果中【{}】表字段【{}】为敏感字段,需要对其映射到对象中的【{}】属性集进行解密", id, resultSetMappingRelation.getTableName(), resultSetMappingRelation.getColumnName(), resultSetMappingRelation.getAliasColumnName());
                        fieldAndDesensitizationModeMap.put(resultSetMappingRelation.getAliasColumnName(), sensitiveWordDatabase.getDesensitizationMode());
                    }
                }
            }
            return fieldAndDesensitizationModeMap;
        } catch (Exception e) {
            throw new DatabaseEncryptException(e);
        }
    }
}