package com.digiwin.dap.middle.ram.filter;

import com.digiwin.dap.middle.ram.constant.I18nError;
import com.digiwin.dap.middle.ram.domain.TargetInfo;
import com.digiwin.dap.middle.ram.domain.TargetInfoContextHolder;
import com.digiwin.dap.middle.ram.service.policy.auth.AuthCheckHandler;
import com.digiwin.dap.middleware.domain.DapEnv;
import com.digiwin.dap.middleware.domain.FilterOrderEnum;
import com.digiwin.dap.middleware.exception.DapException;
import com.digiwin.dap.middleware.util.ExceptionUtils;
import org.springframework.core.Ordered;
import org.springframework.web.filter.OncePerRequestFilter;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

/**
 * AppAuthCheckFilter.java
 *
 * @author fobgo
 * @date 2022/4/11 16:48
 */
public class AuthCheckFilter extends OncePerRequestFilter implements Ordered {

    private final DapEnv dapEnv;
    private final AuthCheckHandler authCheckHandler;

    public AuthCheckFilter(DapEnv dapEnv, AuthCheckHandler authCheckHandler) {
        this.dapEnv = dapEnv;
        this.authCheckHandler = authCheckHandler;
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        final TargetInfo targetInfo = TargetInfoContextHolder.getTargetInfo();
        targetInfo.init();
        try {
            // 自己不检查
            if (request.getRequestURI().startsWith("/api/ram")) {
                filterChain.doFilter(request, response);
                return;
            }

            // 的登陆不在这里检查
            if (request.getRequestURI().startsWith("/api/iam/v2/identity/login")) {
                targetInfo.getAuthResult().setDisabled(true);
            }

            // 应用授权检查
            authCheckHandler.processAuthWithCheck(targetInfo, request);
            filterChain.doFilter(request, response);
        } catch (DapException e) {
            ExceptionUtils.writeUnAuth(request, response, dapEnv.getAppName(), e);
        } catch (Exception e) {
            ExceptionUtils.writeUnAuth(request, response, dapEnv.getAppName(), I18nError.APP_AUTH_CHECK);
        }
    }


    @Override
    public int getOrder() {
        return FilterOrderEnum.APP_CHECK.order();
    }
}
