package com.digiwin.dap.middle.ram.filter;

import com.digiwin.dap.middle.ram.domain.TargetInfo;
import com.digiwin.dap.middle.ram.domain.TargetInfoContextHolder;
import com.digiwin.dap.middle.ram.domain.enums.ResultType;
import com.digiwin.dap.middle.ram.service.PolicyTypeChain;
import com.digiwin.dap.middle.ram.service.policy.filiter.PolicyFilterHandler;
import com.digiwin.dap.middleware.domain.CommonCode;
import com.digiwin.dap.middleware.domain.DapEnv;
import com.digiwin.dap.middleware.domain.FilterOrderEnum;
import com.digiwin.dap.middleware.util.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.Ordered;
import org.springframework.web.filter.OncePerRequestFilter;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.List;

/**
 * 访问控制RAM（Resource Access Management）
 * 管理用户身份与资源访问权限
 *
 * @author fobgo
 * @date 2021/9/29 15:28
 */
public class BeforeAuthenticationFilter extends OncePerRequestFilter implements Ordered {

    private static final Logger logger = LoggerFactory.getLogger(BeforeAuthenticationFilter.class);
    private final DapEnv dapEnv;
    private final List<PolicyFilterHandler> policyHandlers;
    private final boolean obsolete;

    public BeforeAuthenticationFilter(DapEnv dapEnv, List<PolicyFilterHandler> policyHandlers, boolean obsolete) {
        this.dapEnv = dapEnv;
        this.policyHandlers = policyHandlers;
        this.obsolete = obsolete;
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        final TargetInfo targetInfo = TargetInfoContextHolder.getTargetInfo();
        targetInfo.setObsolete(obsolete);
        try {
            PolicyTypeChain policyTypeChain = new PolicyTypeChain(policyHandlers);
            ResultType result = policyTypeChain.doNextStep(targetInfo, request);
            targetInfo.setResultType(result);
        } catch (Exception e) {
            logger.error("RAM before authentication filter handler error", e);
        }

        try {
            if (targetInfo.getResultType() == ResultType.OBSOLETE_DENY) {
                ExceptionUtils.writeUnAuth(request, response, dapEnv.getAppName(), targetInfo.getResultType());
            } else {
                filterChain.doFilter(request, response);
            }
        } finally {
            TargetInfoContextHolder.clearContext();
        }
    }

    @Override
    public int getOrder() {
        return FilterOrderEnum.BEFORE_SECURITY.order();
    }
}
