package com.digiwin.dap.middle.ram.service.policy.auth;

import com.digiwin.dap.middle.ram.domain.AuthResult;
import com.digiwin.dap.middle.ram.domain.TargetInfo;
import com.digiwin.dap.middle.ram.domain.enums.AuthType;
import com.digiwin.dap.middle.ram.service.AuthCheckService;
import com.digiwin.dap.middle.ram.service.PolicyQueryService;
import com.digiwin.dap.middle.ram.util.AuthUtils;
import com.digiwin.dap.middleware.auth.AuthoredSys;
import com.digiwin.dap.middleware.auth.AuthoredUser;
import com.digiwin.dap.middleware.domain.DapEnv;
import org.springframework.core.annotation.AnnotationAwareOrderComparator;

import javax.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.List;

/**
 * CompositeAppAuthCheckHandler.java
 *
 * @author fobgo
 * @date 2022/4/12 17:18
 */
public class AuthCheckHandlerComposite extends AuthCheckHandler {

    private final List<AuthCheckHandler> handlers = new ArrayList<>(5);

    public AuthCheckHandlerComposite(DapEnv dapEnv, AuthCheckService authCheckService, PolicyQueryService policyQueryService) {
        this.handlers.add(new BeforeAuthCheckHandler(dapEnv, policyQueryService));
        this.handlers.add(new NoAuthAuthCheckHandler(policyQueryService));
        this.handlers.add(new DoAuthCheckHandler(authCheckService));
        this.handlers.add(new NoAuthAppAuthCheckHandler(policyQueryService));
        this.handlers.add(new NoAuthUserAuthCheckHandler(policyQueryService));
        AnnotationAwareOrderComparator.sort(this.handlers);
    }

    @Override
    public boolean supports(AuthType authType) {
        return false;
    }

    @Override
    public AuthResult processAuth(TargetInfo targetInfo, HttpServletRequest request) {
        for (AuthCheckHandler handler : handlers) {
            if (handler.supports(targetInfo.getAuthResult().getAuthType())) {
                AuthResult temp = handler.processAuth(targetInfo, request);
                if (temp.getAuthType() == AuthType.AppOk) {
                    return temp;
                }
                targetInfo.setAuthResult(temp);
            }
        }
        return targetInfo.getAuthResult();
    }

    @Override
    public AuthResult processAuthWithCheck(TargetInfo targetInfo, HttpServletRequest request) {
        AuthResult authResult = this.processAuth(targetInfo, request);
        if (authResult.getAuthType() != AuthType.AppOk) {
            AuthUtils.checkAuthResult(targetInfo);
        }
        return authResult;
    }

    @Override
    public AuthResult processAuth(AuthResult authResult, AuthoredUser user, AuthoredSys sys) {
        for (AuthCheckHandler handler : handlers) {
            if (handler.supports(authResult.getAuthType())) {
                AuthResult temp = handler.processAuth(authResult, user, sys);
                if (temp.getAuthType() == AuthType.AppOk) {
                    return temp;
                }
                authResult = temp;
            }
        }
        return authResult;
    }

    @Override
    public AuthResult processAuthWithCheck(AuthResult authResult, AuthoredUser user, AuthoredSys sys) {
        authResult = this.processAuth(authResult, user, sys);
        if (authResult.getAuthType() != AuthType.AppOk) {
            AuthUtils.checkAuthResult(authResult.getAuthType(), user, sys);
        }
        return authResult;
    }
}
