package com.digiwin.dap.middle.ram.service.authentication.handler;

import com.digiwin.dap.middle.ram.domain.PatternVO;
import com.digiwin.dap.middle.ram.domain.Statement;
import com.digiwin.dap.middle.ram.domain.enums.PolicyType;
import com.digiwin.dap.middle.ram.domain.enums.TargetType;
import com.digiwin.dap.middle.ram.service.authentication.AuthCheckHandler;
import com.digiwin.dap.middle.ram.service.core.RamCoreService;
import com.digiwin.dap.middle.ram.util.MatcherUtils;
import com.digiwin.dap.middleware.auth.AuthoredSys;
import com.digiwin.dap.middleware.auth.AuthoredUser;
import com.digiwin.dap.middleware.auth.domain.AuthResult;
import com.digiwin.dap.middleware.auth.domain.AuthType;
import com.digiwin.dap.middleware.auth.domain.RequestInfo;
import org.springframework.core.annotation.Order;

import java.util.List;

/**
 * 不做应用购买检查和用户加入检查<br>
 * 策略类型 {@link PolicyType#NoAuthApp}<br>
 * <ol>
 *     <li>指定应用不检查授权</li>
 *     <li>指定应用的API不检查授权</li>
 * </ol>
 *
 * @author fobgochod
 * @date 2022/4/11 15:04
 */
@Order(15)
public class NoAuthAppAuthCheckHandler extends AuthCheckHandler {

    public NoAuthAppAuthCheckHandler(RamCoreService ramCoreService) {
        this.ramCoreService = ramCoreService;
    }

    @Override
    public boolean supports(AuthType authType) {
        return authType == AuthType.AppNone;
    }

    @Override
    public AuthResult processAuth(AuthResult authResult, AuthoredUser user, AuthoredSys sys, RequestInfo request) {
        List<String> policyIds = ramCoreService.findPolicyId(request.getAppId(), PolicyType.NoAuthApp.name(), TargetType.Sys.name(), sys.getId());
        if (!policyIds.isEmpty()) {
            if (policyIds.contains(PolicyType.NoAuthApp.name())) {
                // 兼容旧，策略ID=NoAuthApp，所有API不检查授权
                authResult.setAuthType(AuthType.AppOk);
            } else {
                // 否则，匹配上的API才不检查授权
                Statement statement = ramCoreService.getPolicyRoute(request.getAppId(), policyIds);
                PatternVO matched = MatcherUtils.matches(request.getMethod(), request.getPath(), statement.getAllow());
                if (matched != null) {
                    authResult.setAuthType(AuthType.AppOk);
                }
            }
        }
        return authResult;
    }
}
