package com.digiwin.dap.middle.ram.service.authentication.handler;

import com.digiwin.dap.middle.ram.domain.enums.PolicyType;
import com.digiwin.dap.middle.ram.domain.enums.TargetType;
import com.digiwin.dap.middle.ram.service.authentication.AuthCheckHandler;
import com.digiwin.dap.middle.ram.service.core.RamCoreService;
import com.digiwin.dap.middleware.auth.AuthoredSys;
import com.digiwin.dap.middleware.auth.AuthoredUser;
import com.digiwin.dap.middleware.auth.domain.AuthResult;
import com.digiwin.dap.middleware.auth.domain.AuthType;
import com.digiwin.dap.middleware.auth.domain.RequestInfo;
import org.springframework.core.annotation.Order;

import java.util.List;

/**
 * 不做用户加入检查，即存在租户应用关系，但是未给用户授权该应用<br>
 * 策略类型 {@link PolicyType#NoAuthUser}<br>
 * <ol>
 *     <li>指定应用不检查用户授权</li>
 *     <li>指定应用租户不检查用户授权</li>
 * </ol>
 *
 * @author fobgochod
 * @date 2022/4/11 15:04
 */
@Order(20)
public class NoAuthUserAuthCheckHandler extends AuthCheckHandler {

    public NoAuthUserAuthCheckHandler(RamCoreService ramCoreService) {
        this.ramCoreService = ramCoreService;
    }

    @Override
    public boolean supports(AuthType authType) {
        return authType == AuthType.AppNoneUser;
    }

    @Override
    public AuthResult processAuth(AuthResult authResult, AuthoredUser user, AuthoredSys sys, RequestInfo request) {
        // 应用层级，该应用所有用户不用检查
        List<String> policyId = ramCoreService.findPolicyId(request.getAppId(), PolicyType.NoAuthUser.name(), TargetType.Sys.name(), sys.getId());
        if (!policyId.isEmpty()) {
            authResult.setAuthType(AuthType.AppOk);
        } else {
            // 租户-应用层级，该租户购买的应用用户不用授权检查
            policyId = ramCoreService.findPolicyId(request.getAppId(), PolicyType.NoAuthUser.name(), TargetType.TenantSys.name(), user.getTenantId() + '-' + sys.getId());
            if (!policyId.isEmpty()) {
                authResult.setAuthType(AuthType.AppOk);
            }
        }
        return authResult;
    }
}
