package com.digiwin.dap.middle.ram.api;

import com.digiwin.dap.middle.ram.domain.GrantVO;
import com.digiwin.dap.middle.ram.domain.enums.PolicyType;
import com.digiwin.dap.middle.ram.domain.page.Page;
import com.digiwin.dap.middle.ram.entity.BaseSid;
import com.digiwin.dap.middle.ram.entity.Policy;
import com.digiwin.dap.middle.ram.service.base.RamGrantPolicyCrudService;
import com.digiwin.dap.middle.ram.service.base.RamPolicyCrudService;
import com.digiwin.dap.middle.ram.service.base.RamPolicyRouteCrudService;
import com.digiwin.dap.middle.ram.service.core.RamCacheService;
import com.digiwin.dap.middleware.domain.StdData;
import com.digiwin.dap.middleware.exception.BusinessException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.stream.Collectors;

/**
 * 策略信息
 *
 * @author fobgochod
 * @date 2021/8/19 9:31
 */
@RequestMapping("/api/ram/v2/policy")
@RestController
public class RamPolicyController {

    @Autowired
    private RamCacheService ramCacheService;
    @Autowired
    private RamPolicyCrudService ramPolicyCrudService;
    @Autowired
    private RamGrantPolicyCrudService ramGrantPolicyCrudService;
    @Autowired
    private RamPolicyRouteCrudService ramPolicyRouteCrudService;

    @PostMapping("/add")
    public StdData<?> add(@RequestBody Policy body) {
        if (PolicyType.isSole(body.getType())) {
            Policy policy = ramPolicyCrudService.findByAppAndType(body.getApp(), body.getType());
            if (policy != null) {
                throw new BusinessException(String.format("[%s]类型策略一个应用只允许建立一个", body.getType()));
            }
        }
        long sid = ramPolicyCrudService.insert(body);
        return StdData.ok(BaseSid.ok(sid));
    }

    @Transactional
    @PostMapping("/del")
    public StdData<?> del(@RequestBody BaseSid body) {
        Policy policy = ramPolicyCrudService.findBySid(body.getSid());
        if (policy == null) {
            return StdData.ok(BaseSid.ok(0));
        }
        List<GrantVO> policyGrants = ramGrantPolicyCrudService.findByPolicySid(body.getSid());
        if (!policyGrants.isEmpty()) {
            String grant = policyGrants.stream().map(o -> o.getType() + "-" + o.getTargetId()).collect(Collectors.joining(","));
            throw new BusinessException(String.format("该策略存在授权信息[%s]，不允许删除", grant));
        }
        ramPolicyCrudService.deleteBySid(body.getSid());
        ramPolicyRouteCrudService.deleteByPolicySid(body.getSid());
        ramCacheService.policyRemove(policy);
        return StdData.ok(BaseSid.ok(body.getSid()));

    }

    @PostMapping("/mod")
    public StdData<?> mod(@RequestBody Policy body) {
        Policy policy = ramPolicyCrudService.findBySid(body.getSid());
        if (policy != null) {
            policy.setName(body.getName());
            ramPolicyCrudService.update(policy);
        }
        return StdData.ok(policy);
    }

    @PostMapping("/get")
    public StdData<?> get(@RequestBody BaseSid body) {
        Policy path = ramPolicyCrudService.findBySid(body.getSid());
        return StdData.ok(path);
    }

    @PostMapping("/find")
    public StdData<?> find(@RequestBody Policy body) {
        return StdData.ok(ramPolicyCrudService.findByCond(body));
    }

    @PostMapping("/query")
    public StdData<?> query(@RequestBody Page<Policy> body) {
        body.setOrderBy("p.create_date desc");
        return StdData.ok(ramPolicyCrudService.findByPage(body));
    }
}
