package com.digiwin.dap.middle.ram.service.core;

import com.digiwin.dap.middle.ram.domain.Statement;
import com.digiwin.dap.middle.ram.domain.condition.HttpMethod;
import com.digiwin.dap.middle.ram.domain.enums.PolicyType;
import com.digiwin.dap.middle.ram.domain.enums.TargetType;
import com.digiwin.dap.middle.ram.service.MappingRegistry;

import java.util.List;

/**
 * 核心API
 * <p>
 * 查询目标、策略、路由之间的关系
 *
 * @author fobgochod
 * @date 2022/6/22 11:43
 */
public interface RamCoreService {

    /**
     * 获取应用的所有路由，用于和请求路由全局匹配<br>
     * {@link PolicyType#Function} 针对rest风格API，找到真正的路由
     *
     * @param appId 应用ID
     */
    MappingRegistry getPattern(String appId);

    /**
     * 查询特殊策略下绑定的路由
     *
     * @param appId      应用ID
     * @param policyType 策略类型 {@link PolicyType#AllowList} {@link PolicyType#None} {@link PolicyType#NoAuth}
     */
    MappingRegistry getPattern(String appId, String policyType);

    /**
     * 查询指定策略的路由
     *
     * @param appId     应用ID
     * @param policyIds 策略ID
     * @return 策略设定的路由
     */
    Statement getPolicyRoute(String appId, List<String> policyIds);

    /**
     * 获取目标的路由<br>
     * 策略类型
     *
     * @param appId      应用ID
     * @param policyType 策略类型 {@link PolicyType#AllowList} {@link PolicyType#Base}
     * @param targetType 目标类型 {@link TargetType}
     * @param targetId   目标ID
     * @return 策略设定的路由
     */
    Statement getPolicyRoute(String appId, String policyType, String targetType, String targetId);

    /**
     * 服务权限<br>
     * 获取配置了指定路由的策略ID<br>
     * 策略{@link PolicyType#Function}的ID即应用、模组、行为、状态全路径 (如：drn:iam:app:DigiwinCloud:djc-cloud-management:djc-mang-service)<br>
     *
     * @param appId      应用ID
     * @param policyType 策略类型 {@link PolicyType#Function}
     * @param method     请求方法 {@link HttpMethod}
     * @param path       请求路由
     * @return 策略ID
     */
    List<String> findPolicyIdByRoute(String appId, String policyType, String method, String path);

    @Deprecated
    List<String> findPolicyIdByRoute(String appId, String method, String path);

    /**
     * 获取目标设定的策略ID
     *
     * @param appId      应用ID
     * @param policyType 策略类型 {@link PolicyType#AllowList} {@link PolicyType#Base} {@link PolicyType#NoAuthApp} {@link PolicyType#NoAuthUser}
     * @param targetType 目标类型 {@link TargetType}
     * @param targetId   目标ID
     * @return 策略ID
     */
    List<String> findPolicyId(String appId, String policyType, String targetType, String targetId);

    /**
     * 获取路由对应的目标ID<br>
     * 策略类型
     *
     * @param appId      应用ID
     * @param policyType 策略类型 {@link PolicyType}
     * @param targetType 目标类型 {@link TargetType}
     * @param method     请求方法 {@link HttpMethod}
     * @param path       请求路由
     * @return 路由对应的目标ID
     */
    List<String> findTargetIdByRoute(String appId, String policyType, String targetType, String method, String path);
}
